local SolarEngineSDK = {}

local sub_lib_version = "1.0.0"

local luaoc = nil
local luaj = nil
local javaProxyClass = "org/cocos2dx/lua/SolarEngineProxy"

local function callAndroidMethod(methodName, params, signature)
    local ok, result = luaj.callStaticMethod(javaProxyClass, methodName, params, signature)
    if ok then
        return result
    else
        print("[SolarEngineSDK] Android " .. methodName .. " call fail")
        return nil
    end
end

local function callAndroidMethodVoid(methodName, params, signature)
    local ok = luaj.callStaticMethod(javaProxyClass, methodName, params, signature)
    if not ok then
        print("[SolarEngineSDK] Android " .. methodName .. " call fail")
    end
end

local function callIOSMethod(methodName, params, signature)
    local ok, ret = luaoc.callStaticMethod("SolarEngineProxy", methodName, params, signature)
    if ok then
        return ret
    else
        print("[SolarEngineSDK] iOS " .. methodName .. " call fail")
        return nil
    end
end

local function callIOSMethodVoid(methodName, params, signature)
    local ok = luaoc.callStaticMethod("SolarEngineProxy", methodName, params, signature)
    if not ok then
        print("[SolarEngineSDK] iOS " .. methodName .. " call fail")
    end
end

local function setiOSCallback(callbackName, callbackFunc, iosMethod, customCallback)
    if callbackFunc and type(callbackFunc) == "function" then
        if customCallback then
            cc.exports[callbackName] = customCallback
        else
            cc.exports[callbackName] = function(data)
                callbackFunc(data)
            end
        end
        callIOSMethodVoid(iosMethod, {func = callbackName}, "(NSDictionary)")
    else
        callIOSMethodVoid(iosMethod, {func = ""}, "(NSDictionary)")
    end
end

local function setAndroidCallback(callbackName, callbackFunc, androidMethod, customCallback)
    if callbackFunc and type(callbackFunc) == "function" then
        if customCallback then
            cc.exports[callbackName] = customCallback
        else
            cc.exports[callbackName] = function(jsonString)
                local result = json.decode(jsonString) or jsonString
                callbackFunc(result)
            end
        end
        callAndroidMethodVoid(androidMethod, {callbackName}, "(Ljava/lang/String;)V")
    else
        callAndroidMethodVoid(androidMethod, {""}, "(Ljava/lang/String;)V")
    end
end


if device.platform == "ios" or device.platform == "mac" then
    local ok, result = pcall(function()
        return require("cocos.cocos2d.luaoc")
    end)
    if ok then
        luaoc = result
    else
        print("[SolarEngineSDK] waring: load fial luaoc :", result)
    end
elseif device.platform == "android" then
    local ok, result = pcall(function()
        return require("cocos.cocos2d.luaj")
    end)
    if ok then
        luaj = result
    else
        print("[SolarEngineSDK] waring: load fial luaj :", result)
    end
else
    print("[SolarEngineSDK] current platform:", device.platform, "- not supported the platform")
end

SolarEngineSDK.PlatformIOS = function()
    if device.platform == "ios" then
        return true        
    end
    return false
end

SolarEngineSDK.PlatformAndroid = function()
    if device.platform == "android" then
        return true        
    end
    return false
end

local function encodeProperties(properties)
    if type(properties) ~= "table" then
        return properties
    end
    for key, value in pairs(properties) do
        if type(value) == "table" then
            properties[key] = json.encode(value)
        end
    end
    return properties
end

SolarEngineSDK.setInitCallback = function(cb)
    if SolarEngineSDK.PlatformIOS() then
        setiOSCallback("__SEInitCallback__", cb, "setInitCallback")
    elseif SolarEngineSDK.PlatformAndroid() then
        local customAndroidCallback = function(codeString)
            local code = tonumber(codeString) or 0
            cb(code)
        end
        setAndroidCallback("__SEInitCallback__", cb, "setInitCallback", customAndroidCallback)
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.setDeeplinkCallback = function(cb)
    if SolarEngineSDK.PlatformIOS() then
        local customIOSCallback = function(jsonString)
            print("[SolarEngineSDK] DeeplinkCallback JSON:", jsonString)
            local simpleResult = json.decode(jsonString)
            if simpleResult then
                cb(simpleResult)
            else
                print("[SolarEngineSDK] parse fail")
                cb(jsonString)
            end
        end
        setiOSCallback("__SEDeeplinkCallback__", cb, "setDeeplinkCallback",customIOSCallback)
    elseif SolarEngineSDK.PlatformAndroid() then
        local customAndroidCallback = function(jsonString)
            print("[SolarEngineSDK] Android  DeeplinkCallback JSON:", jsonString)
            local simpleResult = json.decode(jsonString)
            if simpleResult then
                cb(simpleResult)
            else
                print("[SolarEngineSDK] parse fail")
                cb(jsonString)
            end
        end
        setAndroidCallback("__SEDeeplinkCallback__", cb, "setDeeplinkCallback",customAndroidCallback)
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.setDeferredDeeplinkCallback = function(cb)
    if SolarEngineSDK.PlatformIOS() then
        local customIOSCallback = function(jsonString)
            print("[SolarEngineSDK] DeferredDeeplinkCallback JSON:", jsonString)
            local simpleResult = json.decode(jsonString)
            if simpleResult then
                cb(simpleResult)
            else
                print("[SolarEngineSDK] parse fail")
                cb(jsonString)
            end
        end
        setiOSCallback("__SEDeferredDeeplinkCallback__", cb, "setDeferredDeeplinkCallback",customIOSCallback)
    elseif SolarEngineSDK.PlatformAndroid() then
        local customAndroidCallback = function(jsonString)
            print("[SolarEngineSDK] Android  DeferredDeeplinkCallback JSON:", jsonString)
            local simpleResult = json.decode(jsonString)
            if simpleResult then
                cb(simpleResult)
            else
                print("[SolarEngineSDK] parse fail")
                cb(jsonString)
            end
        end
        setAndroidCallback("__SEDeferredDeeplinkCallback__", cb, "setDeferredDeeplinkCallback",customAndroidCallback)
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end


SolarEngineSDK.setAttributionCallback = function(cb)
    if SolarEngineSDK.PlatformIOS() then
        local customIOSCallback = function(jsonString)
            print("[SolarEngineSDK] AttributionCallback JSON:", jsonString)
            local simpleResult = json.decode(jsonString)
            if simpleResult then
                cb(simpleResult)
            else
                print("[SolarEngineSDK] parse fail")
                cb(jsonString)
            end
        end
        setiOSCallback("__SEAttributionCallback__", cb, "setAttributionCallback", customIOSCallback)
    elseif SolarEngineSDK.PlatformAndroid() then
        local customAndroidCallback = function(jsonString)
            print("[SolarEngineSDK] Android  AttributionCallback JSON:", jsonString)
            local simpleResult = json.decode(jsonString)
            if simpleResult then
                cb(simpleResult)
            else
                print("[SolarEngineSDK] parse fail")
                cb(jsonString)
            end
        end
        setAndroidCallback("__SEAttributionCallback__", cb, "setAttributionCallback", customAndroidCallback)
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end


SolarEngineSDK.preInit = function(appkey)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("preInit", {appkey=appkey})
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("preInit", {appkey}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.initWithConfig = function(config)
    config["sub_lib_version"] = sub_lib_version
    if SolarEngineSDK.PlatformIOS() then
        local args = encodeProperties(config or {})
        callIOSMethodVoid("initWithConfig", args)
    elseif SolarEngineSDK.PlatformAndroid() then
        local json_string = json.encode(config)
        print(json_string)
        callAndroidMethodVoid("initWithConfig", {json_string}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

-- Track Event API
SolarEngineSDK.track = function(eventName, customProperties)
    if SolarEngineSDK.PlatformIOS() then
        local params = {
            eventName = eventName,
            customProperties = customProperties
        }
        callIOSMethodVoid("track", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(customProperties or {})
        callAndroidMethodVoid("trackEvent", {eventName, propertiesJson}, "(Ljava/lang/String;Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.trackFirstEvent = function(eventName, firstCheckId, customProperties)
    if SolarEngineSDK.PlatformIOS() then
        local params = {
            eventName = eventName,
            firstCheckId = firstCheckId,
            customProperties = customProperties
        }
        callIOSMethodVoid("trackFirstEvent", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(customProperties or {})
        callAndroidMethodVoid("trackFirstEvent", {eventName, firstCheckId, propertiesJson}, "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.trackIAP = function(params)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("trackIAP", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(params or {})
        callAndroidMethodVoid("trackIAP", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.trackAdImpression = function(params)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("trackAdImpression", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(params or {})
        callAndroidMethodVoid("trackAdImpression", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.trackAdClick = function(params)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("trackAdClick", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(params or {})
        callAndroidMethodVoid("trackAdClick", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.trackRegister = function(params)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("trackRegister", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(params or {})
        callAndroidMethodVoid("trackRegisterEvent", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.trackLogin = function(params)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("trackLogin", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(params or {})
        callAndroidMethodVoid("trackLoginEvent", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.trackOrder = function(params)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("trackOrder", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(params or {})
        callAndroidMethodVoid("trackOrderEvent", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.trackAppAttr = function(params)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("trackAppAttr", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(params or {})
        callAndroidMethodVoid("trackAttrEvent", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.eventStart = function(eventName)
    if SolarEngineSDK.PlatformIOS() then
        local params = { eventName = eventName }
        callIOSMethodVoid("eventStart", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("eventStart", {eventName}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.eventFinish = function(eventName, properties)
    if SolarEngineSDK.PlatformIOS() then
        local params = {
            eventName = eventName,
            properties = properties
        }
        callIOSMethodVoid("eventFinish", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(properties or {})
        callAndroidMethodVoid("eventFinish", {eventName, propertiesJson}, "(Ljava/lang/String;Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.reportEventImmediately = function()
    if SolarEngineSDK.PlatformIOS() then
        local params = {}
        callIOSMethodVoid("reportEventImmediately", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("reportEventImmediately", {}, "()V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.loginWithAccountID = function(accountId)
    if SolarEngineSDK.PlatformIOS() then
        local params = { accountId = accountId }
        callIOSMethodVoid("loginWithAccountID", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("loginWithAccountID", {accountId}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.logout = function()
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("logout", {}, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("logout", {}, "()V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.getAccountID = function()
    if SolarEngineSDK.PlatformIOS() then
        local ret = callIOSMethod("getAccountID", {}, "(NSDictionary)")
        if ret then
            return ret.accountId
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        local result = callAndroidMethod("getAccountID", {}, "()Ljava/lang/String;")
        return result or ""
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
        return ""
    end
end

SolarEngineSDK.getDistinctId = function()
    if SolarEngineSDK.PlatformIOS() then
        local ret = callIOSMethod("getDistinctId", {}, "(NSDictionary)")
        if ret then
            return ret.distinctId
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        local result = callAndroidMethod("getDistinctId", {}, "()Ljava/lang/String;")
        return result or ""
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
        return ""
    end
end

SolarEngineSDK.setVisitorID = function(visitorId)
    if SolarEngineSDK.PlatformIOS() then
        local params = { visitorId = visitorId }
        callIOSMethodVoid("setVisitorID", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("setVisitorID", {visitorId}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.getVisitorID = function()
    if SolarEngineSDK.PlatformIOS() then
        local ret = callIOSMethod("getVisitorID", {}, "(NSDictionary)")
        if ret then
            return ret.visitorId
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        local result = callAndroidMethod("getVisitorID", {}, "()Ljava/lang/String;")
        return result or ""
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
        return ""
    end
end

SolarEngineSDK.setPresetEvent = function(eventType, properties)
    if SolarEngineSDK.PlatformIOS() then
        local params = {
            eventType = eventType,
            properties = properties
        }
        callIOSMethodVoid("setPresetEvent", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(properties or {})
        callAndroidMethodVoid("setPresetEvent", {eventType, propertiesJson}, "(Ljava/lang/String;Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.setSuperProperties = function(properties)
    if SolarEngineSDK.PlatformIOS() then
        local params = { properties = properties }
        callIOSMethodVoid("setSuperProperties", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(properties or {})
        callAndroidMethodVoid("setSuperProperties", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.unsetSuperProperty = function(key)
    if SolarEngineSDK.PlatformIOS() then
        local params = { key = key }
        callIOSMethodVoid("unsetSuperProperty", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("unsetSuperProperties", {key}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.clearSuperProperties = function()
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("clearSuperProperties", {}, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("clearSuperProperties", {}, "()V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.getPresetProperties = function()
    if SolarEngineSDK.PlatformIOS() then
        local ret = callIOSMethod("getPresetProperties", {}, "(NSDictionary)")
        if ret then
            return ret
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        local result = callAndroidMethod("getPresetProperties", {}, "()Ljava/lang/String;")
        if result then
            -- result 中 ua 会存在/ 解析会出现问题
            result = result:gsub("\\/", "/")
            return json.decode(result)
        end
        return {}
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
        return {}
    end
end

-- User Action API
SolarEngineSDK.userInit = function(properties)
    if SolarEngineSDK.PlatformIOS() then
        local params = { properties = properties }
        callIOSMethodVoid("userInit", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(properties or {})
        callAndroidMethodVoid("userInit", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.userUpdate = function(properties)
    if SolarEngineSDK.PlatformIOS() then
        local params = { properties = properties }
        callIOSMethodVoid("userUpdate", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(properties or {})
        callAndroidMethodVoid("userUpdate", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.userAdd = function(properties)
    if SolarEngineSDK.PlatformIOS() then
        local params = { properties = properties }
        callIOSMethodVoid("userAdd", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(properties or {})
        callAndroidMethodVoid("userAdd", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.userUnset = function(keys)
    if SolarEngineSDK.PlatformIOS() then
        local params = { keys = keys }
        callIOSMethodVoid("userUnset", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local params = { keys = keys }
        local propertiesJson = json.encode(params or {})
        callAndroidMethodVoid("userUnset", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.userAppend = function(properties)
    if SolarEngineSDK.PlatformIOS() then
        local params = { properties = properties }
        callIOSMethodVoid("userAppend", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(properties or {})
        callAndroidMethodVoid("userAppend", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.userDelete = function(deleteType)
    if SolarEngineSDK.PlatformIOS() then
        local params = { deleteType = deleteType }
        callIOSMethodVoid("userDelete", params, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("userDelete", {deleteType or "USER_DELETE"}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.setGDPRArea = function(isGDPRArea)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("setGDPRArea", {isGDPRArea = isGDPRArea}, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("setGDPRArea", {tostring(isGDPRArea)}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.setGaid = function(gaid)
    if SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("setGaid", {gaid}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.setChannel = function(channel)
    if SolarEngineSDK.PlatformAndroid() then
        callAndroidMethodVoid("setChannel", {channel}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.getAttributionData = function()
    if SolarEngineSDK.PlatformIOS() then
        local ret = callIOSMethod("getAttributionData", {}, "(NSDictionary)")
        if ret then
            return ret
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        local result = callAndroidMethod("getAttributionData", {}, "()Ljava/lang/String;")
        if result then
            return json.decode(result)
        end
        return {}
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
        return {}
    end
end

SolarEngineSDK.requestTrackingAuthorization = function(callback)
    if SolarEngineSDK.PlatformIOS() then
        if callback and type(callback) == "function" then
            setiOSCallback("__SETrackingAuthCallback__", callback, "requestTrackingAuthorization")
        end
        callIOSMethodVoid("requestTrackingAuthorization", {
            callback = callback and "__SETrackingAuthCallback__" or ""
        }, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        print("[SolarEngineSDK] only ios support")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.trackAppReEngagement = function(customProperties)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("trackAppReEngagement", {customProperties = customProperties}, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(customProperties or {})
        callAndroidMethodVoid("trackAppReEngagement", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

-- ============================================================================
-- iOS SKAN (SKAdNetwork)
-- ============================================================================

SolarEngineSDK.updatePostbackConversionValue = function(conversionValue, callback)
    if SolarEngineSDK.PlatformIOS() then
        if callback and type(callback) == "function" then
            local customCallback = function(errorCode)
                callback(errorCode)
            end
            setiOSCallback("__SESKANCallback__", callback, "updatePostbackConversionValue", customCallback)
            callIOSMethodVoid("updatePostbackConversionValue", {
                conversionValue = conversionValue,
                callback = "__SESKANCallback__"
            }, "(NSDictionary)")
        else
            callIOSMethodVoid("updatePostbackConversionValue", {
                conversionValue = conversionValue,
                callback = ""
            }, "(NSDictionary)")
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        print("[SolarEngineSDK] only ios support")
        callback(-1)
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.updatePostbackConversionValueWithCoarse = function(fineValue, coarseValue, callback)
    if SolarEngineSDK.PlatformIOS() then
        if callback and type(callback) == "function" then
            setiOSCallback("__SESKANCoarseCallback__", callback, "updatePostbackConversionValueWithCoarse")
            callIOSMethodVoid("updatePostbackConversionValueWithCoarse", {
                fineValue = fineValue,
                coarseValue = coarseValue,
                callback = "__SESKANCoarseCallback__"
            }, "(NSDictionary)")
        else
            callIOSMethodVoid("updatePostbackConversionValueWithCoarse", {
                fineValue = fineValue,
                coarseValue = coarseValue,
                callback = ""
            }, "(NSDictionary)")
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        print("[SolarEngineSDK] only ios support")
        callback(-1)
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.updatePostbackConversionValueWithCoarseAndLock = function(fineValue, coarseValue, lockWindow, callback)
    if SolarEngineSDK.PlatformIOS() then
        if callback and type(callback) == "function" then
            setiOSCallback("__SESKANLockCallback__", callback, "updatePostbackConversionValueWithCoarseAndLock")
            callIOSMethodVoid("updatePostbackConversionValueWithCoarseAndLock", {
                fineValue = fineValue,
                coarseValue = coarseValue,
                lockWindow = lockWindow,
                callback = "__SESKANLockCallback__"
            }, "(NSDictionary)")
        else
            callIOSMethodVoid("updatePostbackConversionValueWithCoarseAndLock", {
                fineValue = fineValue,
                coarseValue = coarseValue,
                lockWindow = lockWindow,
                callback = ""
            }, "(NSDictionary)")
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        print("[SolarEngineSDK] only ios support")
        callback(-1)
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

-- ============================================================================
-- RemoteConfig
-- ============================================================================

SolarEngineSDK.setDefaultConfig = function(configData)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("setDefaultConfig", configData)
    elseif SolarEngineSDK.PlatformAndroid() then
        local configJson = json.encode(configData or {})
        callAndroidMethodVoid("setDefaultConfig", {configJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.setRemoteConfigEventProperties = function(properties)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("setRemoteConfigEventProperties", {properties = properties}, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(properties or {})
        callAndroidMethodVoid("setRemoteConfigEventProperties", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.setRemoteConfigUserProperties = function(properties)
    if SolarEngineSDK.PlatformIOS() then
        callIOSMethodVoid("setRemoteConfigUserProperties", {properties = properties}, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        local propertiesJson = json.encode(properties or {})
        callAndroidMethodVoid("setRemoteConfigUserProperties", {propertiesJson}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.fastFetchRemoteConfig = function(key)
    if SolarEngineSDK.PlatformIOS() then
        local ret = callIOSMethod("fastFetchRemoteConfig", {key = key}, "(NSDictionary)")
        if ret then
            return ret
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        local result = callAndroidMethod("fastFetchRemoteConfig", {key}, "(Ljava/lang/String;)Ljava/lang/String;")
        return result or ""
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
        return ""
    end
end

SolarEngineSDK.fastFetchAllRemoteConfig = function()
    if SolarEngineSDK.PlatformIOS() then
        local ret = callIOSMethod("fastFetchAllRemoteConfig", {}, "(NSDictionary)")
        if ret then
            return ret
        end
    elseif SolarEngineSDK.PlatformAndroid() then
        local result = callAndroidMethod("fastFetchAllRemoteConfig", {}, "()Ljava/lang/String;")
        if result then
            return json.decode(result)
        end
        return {}
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
        return {}
    end
end

SolarEngineSDK.asyncFetchRemoteConfig = function(key, callback)
    if SolarEngineSDK.PlatformIOS() then
        if callback and type(callback) == "function" then
            local customIOSCallback = function(jsonString)
                local simpleResult = json.decode(jsonString)
                if simpleResult then
                    callback(simpleResult)
                else
                    callback(jsonString)
                end
            end
            setiOSCallback("__SERemoteConfigCallback__", callback, "asyncFetchRemoteConfig", customIOSCallback)
        end
        callIOSMethodVoid("asyncFetchRemoteConfig", {
            key = key,
            callback = callback and "__SERemoteConfigCallback__" or ""
        }, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        if callback and type(callback) == "function" then
            local customAndroidCallback = function(jsonString)
                local simpleResult = json.decode(jsonString)
                if simpleResult then
                    callback(simpleResult)
                else
                    callback(jsonString)
                end
            end
            setAndroidCallback("__SERemoteConfigCallback__", callback, "setRemoteConfigCallback", customAndroidCallback)
        end
        callAndroidMethodVoid("asyncFetchRemoteConfig", {key or ""}, "(Ljava/lang/String;)V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

SolarEngineSDK.asyncFetchAllRemoteConfig = function(callback)
    if SolarEngineSDK.PlatformIOS() then
        if callback and type(callback) == "function" then
            local customIOSCallback = function(jsonString)
                local simpleResult = json.decode(jsonString)
                if simpleResult then
                    callback(simpleResult)
                else
                    callback(jsonString)
                end
            end
            setiOSCallback("__SERemoteConfigAllCallback__", callback, "asyncFetchAllRemoteConfig", customIOSCallback)
        end
        callIOSMethodVoid("asyncFetchAllRemoteConfig", {
            callback = callback and "__SERemoteConfigAllCallback__" or ""
        }, "(NSDictionary)")
    elseif SolarEngineSDK.PlatformAndroid() then
        if callback and type(callback) == "function" then
            local customAndroidCallback = function(jsonString)
                local simpleResult = json.decode(jsonString)
                if simpleResult then
                    callback(simpleResult)
                else
                    callback(jsonString)
                end
            end
            setAndroidCallback("__SERemoteConfigAllCallback__", callback, "setRemoteConfigAllCallback", customAndroidCallback)
        end
        callAndroidMethodVoid("asyncFetchAllRemoteConfig", {}, "()V")
    else
        print("[SolarEngineSDK] not supported the platform: ", device.platform)
    end
end

return SolarEngineSDK


