
#import "SolarEngineProxy.h"

#import <SolarEngineSDK/SolarEngineSDK.h>
#import <SolarEngineSDK/SEEventConstants.h>
#import <SolarEngineSDK/SEConfig.h>
#import <SolarEngineSDK/SERemoteConfigManager.h>
#import <Foundation/Foundation.h>
#if  __has_include(<SolarEngineSDK/SESDKForCN.h>)
    #import <SolarEngineSDK/SESDKForCN.h>
#endif
#if  __has_include(<SolarEngineSDK/SESDKForUS.h>)
    #import <SolarEngineSDK/SESDKForUS.h>
#endif

#include "scripting/lua-bindings/manual/CCLuaEngine.h"
#include "scripting/lua-bindings/manual/lua_module_register.h"
#include "platform/ios/CCLuaObjcBridge.h"

@interface SEWrapperManager : NSObject

@property (nonatomic,copy)NSString *sub_lib_version;
@property (nonatomic,copy)NSString *sdk_type;

+ (SEWrapperManager *)sharedInstance;

@end
@implementation SEWrapperManager

+ (SEWrapperManager *)sharedInstance {
    static SEWrapperManager * instance = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        instance = [[self alloc] init];
    });
    return instance;
}

- (instancetype)init
{
    self = [super init];
    if (self) {
        _sdk_type = @"cocos2d-lua";
    }
    return self;
}

@end

@interface NSDictionary (SafeMode)
- (id)smValueForKey:(NSString *)key;
- (NSString *)toJsonString;

@end
@implementation NSDictionary (SafeMode)
- (id)smValueForKey:(NSString *)key {
    if (key!=nil) {
        return [self valueForKey:key];
    } else {
        return nil;
    }
}

- (NSString *)toJsonString {
    @try {
        NSData *data = [NSJSONSerialization dataWithJSONObject:self options:0 error:0];
        NSString *string = [[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding];
        return  string;
    } @catch (NSException *exception) {
        
    }
    return @"{}";
}
@end

@interface NSString (SafeMode)
- (NSString *)toDictionary;
@end
@implementation NSString (SafeMode)
- (NSDictionary *)toDictionary {
    @try {
        NSData *data = [self dataUsingEncoding:NSUTF8StringEncoding];
        NSDictionary *dict = [NSJSONSerialization JSONObjectWithData:data options:0 error:0];
        return  dict;
    } @catch (NSException *exception) {
        
    }
    return @{};
}
@end

@interface SolarEngineProxy ()

@end

@implementation SolarEngineProxy

static NSString *sSEInitCallbackName = nil;
static NSString *sSEDeeplinkCallbackName = nil;
static NSString *sSEDeferredDeeplinkCallbackName = nil;
static NSString *sSEAttributionCallbackName = nil;

+ (void)preInit:(NSDictionary *)params {
    NSLog(@"[lua] preInit called : %@",params);

    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] setInitCallback called params class error, params: %@",params);
        return;
    }
    
    NSString *appkey = params[@"appkey"];
    [[SolarEngineSDK sharedInstance] preInitWithAppKey:appkey];
}

// 设置 Lua 回调函数名（通过 Lua 传入 { func = "GlobalFuncName" }）
+ (void)setInitCallback:(NSDictionary *)params {
    
    NSLog(@"[SolarEngineSDK] setInitCallback : %@", params);

    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] setInitCallback called params class error, params: %@",params);
        return;
    }
    
    NSString *func = params[@"func"];
    sSEInitCallbackName = func.length > 0 ? [func copy] : nil;
}
+ (void)setAttributionCallback:(NSDictionary *)params {
    
    NSLog(@"[SolarEngineSDK] setAttributionCallbackName : %@", params);

    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] sSEAttributionCallbackName called params class error, params: %@",params);
        return;
    }
    
    NSString *func = params[@"func"];
    sSEAttributionCallbackName = func.length > 0 ? [func copy] : nil;
}
+ (void)setDeeplinkCallback:(NSDictionary *)params {
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] setDeeplinkCallback called params class error, params: %@",params);
        return;
    }
    
    NSString *func = params[@"func"];
    sSEDeeplinkCallbackName = func.length > 0 ? [func copy] : nil;
}
+ (void)setDeferredDeeplinkCallback:(NSDictionary *)params {
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] setDeferredDeeplinkCallback called params class error, params: %@",params);
        return;
    }
    
    NSString *func = params[@"func"];
    sSEDeferredDeeplinkCallbackName = func.length > 0 ? [func copy] : nil;
}
+ (void)initWithConfig:(NSDictionary *)params {
 
    NSLog(@"[lua] initWithConfig called : %@",params);
    
    NSDictionary *seDict = nil;
    if ([params isKindOfClass:[NSDictionary class]]) {
        seDict = params;
    } else {
        NSLog(@"[lua] initWithConfig called params class error.");
    }
    
    NSString *_appKey = seDict[@"appKey"];

    SEConfig *config = [[SEConfig alloc] init];
    config.isDebugModel = [seDict[@"isDebugModel"] boolValue];
    config.logEnabled = [seDict[@"logEnabled"] boolValue];
    config.setCoppaEnabled = [seDict[@"isCoppaEnabled"] boolValue];
    config.setKidsAppEnabled = [seDict[@"isKidsAppEnabled"] boolValue];
    config.enable2GReporting = [seDict[@"isEnable2GReporting"] boolValue];
    config.isGDPRArea = [seDict[@"isGDPRArea"] boolValue];
    config.attAuthorizationWaitingInterval = [seDict[@"attAuthorizationWaitingInterval"] intValue];
    config.enableDeferredDeeplink = [seDict[@"deferredDeeplinkEnable"] boolValue];

#if  __has_include(<SolarEngineSDK/SESDKForCN.h>)
        if (seDict[@"caid"]) {
            config.caid = seDict[@"caid"];
        }
#endif
    
#if  __has_include(<SolarEngineSDK/SESDKForUS.h>)
        if (seDict[@"odmInfoEnable"]) {
            config.enableODMInfo = [seDict[@"odmInfoEnable"] boolValue];
        }
#endif

    NSString *sub_lib_version = seDict[@"sub_lib_version"];
    if ([sub_lib_version isKindOfClass:[NSString class]]) {
        [SEWrapperManager sharedInstance].sub_lib_version = sub_lib_version;
    }

    NSString *sdk_type = seDict[@"sdk_type"];
    if ([sdk_type isKindOfClass:[NSString class]]) {
        [SEWrapperManager sharedInstance].sdk_type = sdk_type;
    }
    
    NSDictionary *rcDict = seDict[@"remoteConfig"];
    
    if ([rcDict isKindOfClass:[NSString class]]) {
        NSData *data = [(NSString*)rcDict dataUsingEncoding:NSUTF8StringEncoding];
        rcDict = [NSJSONSerialization JSONObjectWithData:data options:0 error:0];
    }
    
    if ([rcDict isKindOfClass:[NSDictionary class]]) {
        
        SERemoteConfig *remoteConfig = [[SERemoteConfig alloc] init];
        remoteConfig.enable = [rcDict[@"enable"] boolValue];
        remoteConfig.logEnabled = config.logEnabled;
        
        remoteConfig.mergeType = (SERCMergeType)[rcDict[@"mergeType"] integerValue];

        remoteConfig.customIDProperties = rcDict[@"customIDProperties"];
        remoteConfig.customIDUserProperties = rcDict[@"customIDUserProperties"];
        remoteConfig.customIDEventProperties = rcDict[@"customIDEventProperties"];
        
        config.remoteConfig = remoteConfig;
    }
    
    NSDictionary *customDomainDict = seDict[@"customDomain"];
    if ([customDomainDict isKindOfClass:[NSDictionary class]]) {
        
        SECustomDomain *customDomain = [[SECustomDomain alloc] init];
        customDomain.enable             = [customDomainDict[@"enable"] boolValue];
        customDomain.receiverDomain     = customDomainDict[@"receiverDomain"];
        customDomain.ruleDomain         = customDomainDict[@"ruleDomain"];
        customDomain.ruleTcpHost        = customDomainDict[@"tcpRuleHost"];
        customDomain.receiverTcpHost    = customDomainDict[@"tcpReceiverHost"];
        customDomain.gatewayTcpHost     = customDomainDict[@"tcpGatewayHost"];
        config.customDomain = customDomain;
    }
    
    [[SolarEngineSDK sharedInstance] setInitCompletedCallback:^(int code) {
        NSLog(@"[SolarEngineSDK] Init completed with code: %d", code);
        [self callLuaWithMethod:sSEInitCallbackName withIntParam:code];
    }];
    
    [[SolarEngineSDK sharedInstance] setAttributionCallback:^(int code, NSDictionary * _Nullable attributionData) {
        NSMutableDictionary *dict = [NSMutableDictionary dictionary];
        if (code == 0){
            if (attributionData) {
                [dict setObject:attributionData forKey:@"data"];
                [dict setObject:[NSNumber numberWithInt:0] forKey:@"code"];
            } else {
                [dict setObject:[NSNumber numberWithInt:-1] forKey:@"code"];
            }
        } else {
            [dict setObject:[NSNumber numberWithInt:code] forKey:@"code"];
        }
        
        NSLog(@"[SolarEngineSDK] Attribution completed with code: %d", code);
        NSString *msg = [dict toJsonString];
        [self callLuaWithMethod:sSEAttributionCallbackName withStringParam:msg];
    }];
    
    [[SolarEngineSDK sharedInstance] setDeepLinkCallback:^(int code, SEDeeplinkInfo * _Nullable deeplinkInfo) {
        
        NSMutableDictionary *deeplinkData = [NSMutableDictionary dictionary];
        if (code == 0){
            if (deeplinkInfo.from) {
                [deeplinkData setObject:deeplinkInfo.from forKey:@"from"];
            }
            if (deeplinkInfo.sedpLink) {
                [deeplinkData setObject:deeplinkInfo.sedpLink forKey:@"sedpLink"];
            }
            if (deeplinkInfo.turlId) {
                [deeplinkData setObject:deeplinkInfo.turlId forKey:@"turlId"];
            }
            if (deeplinkInfo.baseUrl) {
                [deeplinkData setObject:deeplinkInfo.baseUrl forKey:@"baseUrl"];
            }
            if (deeplinkInfo.url) {
                [deeplinkData setObject:deeplinkInfo.url forKey:@"url"];
            }
            if (deeplinkInfo.customParams) {
                [deeplinkData setObject:deeplinkInfo.customParams forKey:@"customParams"];
            }
        }

        NSMutableDictionary *dict = [NSMutableDictionary dictionary];
        [dict setObject:deeplinkData forKey:@"data"];
        [dict setObject:[NSNumber numberWithInt:code] forKey:@"code"];
        
        NSString *msg = [dict toJsonString];
        [self callLuaWithMethod:sSEDeeplinkCallbackName withStringParam:msg];
    }];
    
    [[SolarEngineSDK sharedInstance] setDeferredDeepLinkCallbackWithSuccess:^(SEDeferredDeeplinkInfo * _Nullable deeplinkInfo) {
        
        NSMutableDictionary *deeplinkData = [NSMutableDictionary dictionary];
        if (deeplinkInfo.sedpLink) {
            [deeplinkData setObject:deeplinkInfo.sedpLink forKey:@"sedpLink"];
        }
        if (deeplinkInfo.turlId) {
            [deeplinkData setObject:deeplinkInfo.turlId forKey:@"turlId"];
        }
        if (deeplinkInfo.sedpUrlscheme) {
            [deeplinkData setObject:deeplinkInfo.sedpUrlscheme forKey:@"sedpUrlscheme"];
        }

        NSMutableDictionary *dict = [NSMutableDictionary dictionary];
        [dict setObject:deeplinkData forKey:@"data"];
        [dict setObject:[NSNumber numberWithInt:0] forKey:@"code"];
        
        NSString *msg = [dict toJsonString];
        [self callLuaWithMethod:sSEDeeplinkCallbackName withStringParam:msg];
        
    } fail:^(NSError * _Nullable error) {
        NSMutableDictionary *deeplinkData = [NSMutableDictionary dictionary];
        NSMutableDictionary *dict = [NSMutableDictionary dictionary];
        [dict setObject:[NSNumber numberWithInt:(int)error.code] forKey:@"code"];
        NSString *msg = [dict toJsonString];
        [self callLuaWithMethod:sSEDeferredDeeplinkCallbackName withStringParam:msg];
    }];
    
    [[SolarEngineSDK sharedInstance] startWithAppKey:_appKey config:config];
}

+ (void)callLuaWithMethod:(NSString *)method withIntParam:(int)index {
    
    if (method.length > 0) {
        cocos2d::LuaEngine *engine = cocos2d::LuaEngine::getInstance();
        if (engine) {
            cocos2d::LuaStack *stack = engine->getLuaStack();
            if (stack) {
                lua_getglobal(stack->getLuaState(), [method UTF8String]);
                if (lua_isfunction(stack->getLuaState(), -1)) {
                    lua_pushinteger(stack->getLuaState(), (lua_Integer)index);
                    stack->executeFunction(1);
                } else {
                    NSLog(@"[SESDK], callLuaWithMethod error: functions not callable types !");
                    lua_pop(stack->getLuaState(), 1);
                }
            }
        } else {
            NSLog(@"[SESDK], callLuaWithMethod error: not found engine !");
        }
    } else {
        NSLog(@"[SESDK], callLuaWithMethod error: not found method : %@",method);
    }
}
+ (void)callLuaWithMethod:(NSString *)method withStringParam:(NSString *)string {
    
    if (method.length > 0) {
        cocos2d::LuaEngine *engine = cocos2d::LuaEngine::getInstance();
        if (engine) {
            cocos2d::LuaStack *stack = engine->getLuaStack();
            if (stack) {
                lua_getglobal(stack->getLuaState(), [method UTF8String]);
                if (lua_isfunction(stack->getLuaState(), -1)) {
                    lua_pushstring(stack->getLuaState(), [string UTF8String]);
                    stack->executeFunction(1);
                } else {
                    NSLog(@"[SESDK], callLuaWithMethod error: functions not callable types !");
                    lua_pop(stack->getLuaState(), 1);
                }
            }
        } else {
            NSLog(@"[SESDK], callLuaWithMethod error: not found engine !");
        }
    } else {
        NSLog(@"[SESDK], callLuaWithMethod error: not found method : %@",method);
    }
}

#pragma mark - 事件追踪

+ (void)track:(NSDictionary *)params {
    NSLog(@"[lua] track called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] track called params class error.");
        return;
    }
    
    NSString *eventName = params[@"eventName"];
    NSDictionary *customProperties = params[@"customProperties"];
    
    [[SolarEngineSDK sharedInstance] track:eventName withProperties:customProperties];
}

+ (void)trackIAP:(NSDictionary *)params {
    NSLog(@"[lua] trackIAP called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] trackIAP called params class error.");
        return;
    }
    
    // 创建 IAP 事件属性对象
    SEIAPEventAttribute *attribute = [[SEIAPEventAttribute alloc] init];
    attribute.productID = params[@"productId"];
    attribute.productName = params[@"productName"];
    attribute.productCount = [params[@"productCount"] integerValue];
    attribute.orderId = params[@"orderId"];
    attribute.payAmount = [params[@"payAmount"] doubleValue];
    attribute.currencyType = params[@"currency"];
    attribute.payType = params[@"payType"];
    attribute.payStatus = (SolarEngineIAPStatus)[params[@"payStatus"] integerValue];
    attribute.failReason = params[@"failReason"];
    attribute.customProperties = params[@"customProperties"];
    
    [[SolarEngineSDK sharedInstance] trackIAPWithAttributes:attribute];
}

+ (void)trackAdImpression:(NSDictionary *)params {
    NSLog(@"[lua] trackAdImpression called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] trackAdImpression called params class error.");
        return;
    }
    
    SEAdImpressionEventAttribute *attribute = [[SEAdImpressionEventAttribute alloc] init];
    attribute.adType = (SolarEngineAdType)[params[@"adType"] integerValue];
    attribute.adNetworkPlatform = params[@"adNetworkPlatform"];
    attribute.adNetworkAppID = params[@"adNetworkAppId"];
    attribute.adNetworkPlacementID = params[@"adNetworkPlacementId"];
    attribute.ecpm = [params[@"ecpm"] doubleValue];
    attribute.currency = params[@"currency"];
    attribute.mediationPlatform = params[@"mediationPlatform"];
    attribute.rendered = [params[@"rendered"] boolValue];
    attribute.customProperties = params[@"customProperties"];
    
    [[SolarEngineSDK sharedInstance] trackAdImpressionWithAttributes:attribute];
}

+ (void)trackAdClick:(NSDictionary *)params {
    NSLog(@"[lua] trackAdClick called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] trackAdClick called params class error.");
        return;
    }
    
    SEAdClickEventAttribute *attribute = [[SEAdClickEventAttribute alloc] init];
    attribute.adType = (SolarEngineAdType)[params[@"adType"] integerValue];
    attribute.adNetworkPlatform = params[@"adNetworkPlatform"];
    attribute.adNetworkPlacementID = params[@"adNetworkPlacementId"];
    attribute.mediationPlatform = params[@"mediationPlatform"];
    attribute.customProperties = params[@"customProperties"];
    
    [[SolarEngineSDK sharedInstance] trackAdClickWithAttributes:attribute];
}

+ (void)trackAppAttr:(NSDictionary *)params {
    NSLog(@"[lua] trackAppAttr called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] trackAppAttr called params class error.");
        return;
    }
    
    SEAppAttrEventAttribute *attribute = [[SEAppAttrEventAttribute alloc] init];
    attribute.adNetwork = params[@"adNetwork"];
    attribute.subChannel = params[@"subChannel"];
    attribute.adAccountID = params[@"adAccountId"];
    attribute.adAccountName = params[@"adAccountName"];
    attribute.adCampaignID = params[@"adCampaignId"];
    attribute.adCampaignName = params[@"adCampaignName"];
    attribute.adOfferID = params[@"adOfferId"];
    attribute.adOfferName = params[@"adOfferName"];
    attribute.adCreativeID = params[@"adCreativeId"];
    attribute.adCreativeName = params[@"adCreativeName"];
    attribute.attributionPlatform = params[@"attributionPlatform"];
    attribute.customProperties = params[@"customProperties"];
    
    [[SolarEngineSDK sharedInstance] trackAppAttrWithAttributes:attribute];
}

+ (void)trackRegister:(NSDictionary *)params {
    NSLog(@"[lua] trackRegister called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] trackRegister called params class error.");
        return;
    }
    
    SERegisterEventAttribute *attribute = [[SERegisterEventAttribute alloc] init];
    attribute.registerType = params[@"registerType"];
    attribute.registerStatus = params[@"registerStatus"];
    attribute.customProperties = params[@"customProperties"];
    
    [[SolarEngineSDK sharedInstance] trackRegisterWithAttributes:attribute];
}

+ (void)trackLogin:(NSDictionary *)params {
    NSLog(@"[lua] trackLogin called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] trackLogin called params class error.");
        return;
    }
    
    SELoginEventAttribute *attribute = [[SELoginEventAttribute alloc] init];
    attribute.loginType = params[@"loginType"];
    attribute.loginStatus = params[@"loginStatus"];
    attribute.customProperties = params[@"customProperties"];
    
    [[SolarEngineSDK sharedInstance] trackLoginWithAttributes:attribute];
}

+ (void)trackOrder:(NSDictionary *)params {
    NSLog(@"[lua] trackOrder called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] trackOrder called params class error.");
        return;
    }
    
    SEOrderEventAttribute *attribute = [[SEOrderEventAttribute alloc] init];
    attribute.orderID = params[@"orderId"];
    attribute.payAmount = [params[@"payAmount"] doubleValue];
    attribute.currencyType = params[@"currency"];
    attribute.payType = params[@"payType"];
    attribute.status = params[@"status"];
    attribute.customProperties = params[@"customProperties"];
    
    [[SolarEngineSDK sharedInstance] trackOrderWithAttributes:attribute];
}

+ (void)trackFirstEvent:(NSDictionary *)params {
    NSLog(@"[lua] trackFirstEvent called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] trackFirstEvent called params class error.");
        return;
    }
    
    NSString *firstCheckId = params[@"firstCheckId"];
    NSString *eventName = params[@"eventName"];
    NSDictionary *options  = params[@"customProperties"];
    if ([options isKindOfClass:[NSString class]]) {
        NSData *data = [(NSString*)options dataUsingEncoding:NSUTF8StringEncoding];
        options = [NSJSONSerialization JSONObjectWithData:data options:0 error:0];
    }
    
    if ([eventName isEqualToString:@"_appReg"]) {
        SERegisterEventAttribute *attribute = [[SERegisterEventAttribute alloc] init];
        attribute.registerType = options[@"registerType"];
        attribute.registerStatus = options[@"registerStatus"];
        attribute.customProperties = options[@"customProperties"];
        [[SolarEngineSDK sharedInstance] trackFirstEvent:attribute];
    } else {
        if ([eventName hasPrefix:@"_"]) {
            NSLog(@"[lua] Error: trackFirstEvent api only support _appReg event and custom event.");
        }
        SECustomEventAttribute *attribute = [[SECustomEventAttribute alloc] init];
        attribute.eventName = eventName;
        attribute.firstCheckId = firstCheckId;
        attribute.customProperties = options[@"customProperties"];
        [[SolarEngineSDK sharedInstance] trackFirstEvent:attribute];
    }
}

+ (void)eventStart:(NSDictionary *)params {
    NSLog(@"[lua] eventStart called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] eventStart called params class error.");
        return;
    }
    
    NSString *eventName = params[@"eventName"];
    [[SolarEngineSDK sharedInstance] eventStart:eventName];
}

+ (void)eventFinish:(NSDictionary *)params {
    NSLog(@"[lua] eventFinish called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] eventFinish called params class error.");
        return;
    }
    
    NSString *eventName = params[@"eventName"];
    NSDictionary *properties = params[@"properties"];
    
    [[SolarEngineSDK sharedInstance] eventFinish:eventName properties:properties];
}

+ (void)reportEventImmediately:(NSDictionary *)params {
    NSLog(@"[lua] reportEventImmediately called");
    
    [[SolarEngineSDK sharedInstance] reportEventImmediately];
}

#pragma mark - 用户管理

+ (void)loginWithAccountID:(NSDictionary *)params {
    NSLog(@"[lua] loginWithAccountID called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] loginWithAccountID called params class error.");
        return;
    }
    
    NSString *accountId = params[@"accountId"];
    [[SolarEngineSDK sharedInstance] loginWithAccountID:accountId];
}

+ (void)logout:(NSDictionary *)params {
    NSLog(@"[lua] logout called");
    
    [[SolarEngineSDK sharedInstance] logout];
}

+ (NSDictionary *)getAccountID:(NSDictionary *)params {
    NSLog(@"[lua] getAccountID called");
    
    NSString *accountId = [[SolarEngineSDK sharedInstance] accountID];
    return [NSDictionary dictionaryWithObjectsAndKeys:accountId, @"accountId", nil];
}

+ (NSDictionary *)getDistinctId:(NSDictionary *)params {
    NSLog(@"[lua] getDistinctId called");
    
    NSString *distinctId = [[SolarEngineSDK sharedInstance] getDistinctId];
    return [NSDictionary dictionaryWithObjectsAndKeys:distinctId, @"distinctId", nil];
}

#pragma mark - 访客 ID

+ (void)setVisitorID:(NSDictionary *)params {
    NSLog(@"[lua] setVisitorID called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] setVisitorID called params class error.");
        return;
    }
    
    NSString *visitorId = params[@"visitorId"];
    [[SolarEngineSDK sharedInstance] setVisitorID:visitorId];
}

+ (NSDictionary *)getVisitorID:(NSDictionary *)params {
    
    NSString *visitorId = [[SolarEngineSDK sharedInstance] visitorID];
    NSLog(@"[lua] getVisitorID result: %@", visitorId);
    
    return [NSDictionary dictionaryWithObjectsAndKeys:visitorId, @"visitorId", nil];
}

#pragma mark - 属性管理

+ (void)setSuperProperties:(NSDictionary *)params {
    NSLog(@"[lua] setSuperProperties called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] setSuperProperties called params class error.");
        return;
    }
    
    NSDictionary *properties = params[@"properties"];
    [[SolarEngineSDK sharedInstance] setSuperProperties:properties];
}

+ (void)unsetSuperProperty:(NSDictionary *)params {
    NSLog(@"[lua] unsetSuperProperty called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] unsetSuperProperty called params class error.");
        return;
    }
    
    NSString *key = params[@"key"];
    [[SolarEngineSDK sharedInstance] unsetSuperProperty:key];
}

+ (void)clearSuperProperties:(NSDictionary *)params {
    NSLog(@"[lua] clearSuperProperties called");
    
    [[SolarEngineSDK sharedInstance] clearSuperProperties];
}

+ (NSDictionary *)getPresetProperties:(NSDictionary *)params {
    
    NSDictionary *properties = [[SolarEngineSDK sharedInstance] getPresetProperties];
    NSLog(@"[lua] getPresetProperties result: %@", properties);
    
    // 直接返回属性字典，不需要包装
    return properties ?: @{};
}

+ (void)setPresetEvent:(NSDictionary *)params {
    NSLog(@"[lua] setPresetEvent called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] setPresetEvent called params class error.");
        return;
    }
    
    NSInteger eventType = [params[@"eventType"] integerValue];
    NSDictionary *properties = params[@"properties"];
    
    [[SolarEngineSDK sharedInstance] setPresetEvent:(SEPresetEventType)eventType withProperties:properties];
}

#pragma mark - 用户属性

+ (void)userInit:(NSDictionary *)params {
    NSLog(@"[lua] userInit called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] userInit called params class error.");
        return;
    }
    
    NSDictionary *properties = params[@"properties"];
    [[SolarEngineSDK sharedInstance] userInit:properties];
}

+ (void)userUpdate:(NSDictionary *)params {
    NSLog(@"[lua] userUpdate called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] userUpdate called params class error.");
        return;
    }
    
    NSDictionary *properties = params[@"properties"];
    [[SolarEngineSDK sharedInstance] userUpdate:properties];
}

+ (void)userAdd:(NSDictionary *)params {
    NSLog(@"[lua] userAdd called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] userAdd called params class error.");
        return;
    }
    
    NSDictionary *properties = params[@"properties"];
    [[SolarEngineSDK sharedInstance] userAdd:properties];
}

+ (void)userUnset:(NSDictionary *)params {
    NSLog(@"[lua] userUnset called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] userUnset called params class error.");
        return;
    }
    
    NSArray *keys = params[@"keys"];
    [[SolarEngineSDK sharedInstance] userUnset:keys];
}

+ (void)userAppend:(NSDictionary *)params {
    NSLog(@"[lua] userAppend called : %@", params);
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] userAppend called params class error.");
        return;
    }
    
    NSDictionary *properties = params[@"properties"];
    [[SolarEngineSDK sharedInstance] userAppend:properties];
}

+ (void)userDelete:(NSDictionary *)params {
    
    if (![params isKindOfClass:[NSDictionary class]]) {
        NSLog(@"[lua] userDelete called params class error.");
        return;
    }
    
    NSString *deleteType = params[@"deleteType"];
    if (![deleteType isKindOfClass:[NSString class]]) {
        NSLog(@"[lua] userDelete called deleteType class error.");
    }
    
    if ([deleteType isEqualToString:@"byAccountId"]) {
        [[SolarEngineSDK sharedInstance] userDelete:SEUserDeleteTypeByAccountId];
    } else if ([deleteType isEqualToString:@"byVisitorId"]) {
        [[SolarEngineSDK sharedInstance] userDelete:SEUserDeleteTypeByVisitorId];
    } else {
        NSLog(@"[lua] userDelete called deleteType fail : %@",deleteType);
    }
    
}

#pragma mark - 工具方法

+ (void)setGDPRArea:(NSDictionary *)params {
    
    BOOL isGDPRArea = [params[@"isGDPRArea"] boolValue];
    [[SolarEngineSDK sharedInstance] setGDPRArea:isGDPRArea];
    NSLog(@"[lua] setGDPRArea result: %@", isGDPRArea ? @"YES" : @"NO");
}

+ (NSDictionary *)getAttributionData:(NSDictionary *)params {
    
    NSDictionary *attributionData = [[SolarEngineSDK sharedInstance] getAttributionData];
    NSLog(@"[lua] getAttributionData result: %@", attributionData);
    
    // 直接返回归因数据，不需要包装
    return attributionData ?: @{};
}

+ (void)requestTrackingAuthorization:(NSDictionary *)params {
    NSLog(@"[lua] requestTrackingAuthorization called");
    
    NSString *callbackFunc = params[@"callback"];
    if (callbackFunc && [callbackFunc length] > 0) {
        [[SolarEngineSDK sharedInstance] requestTrackingAuthorizationWithCompletionHandler:^(NSUInteger status) {
            NSLog(@"[lua] requestTrackingAuthorization result: %lu", (unsigned long)status);
            
            // 回调到 Lua
            [SolarEngineProxy callLuaWithMethod:callbackFunc withIntParam:(int)status];
        }];
    } else {
        // 创建一个空的回调
        [[SolarEngineSDK sharedInstance] requestTrackingAuthorizationWithCompletionHandler:^(NSUInteger status) {
            NSLog(@"[lua] requestTrackingAuthorization completed without callback: %lu", (unsigned long)status);
        }];
    }
}

+ (void)trackAppReEngagement:(NSDictionary *)params {
    
    NSDictionary *customProperties = params[@"customProperties"];
    [[SolarEngineSDK sharedInstance] trackAppReEngagement:customProperties];
}

#pragma mark - SKAN

+ (void)updatePostbackConversionValue:(NSDictionary *)params {
    
    NSInteger conversionValue = [params[@"conversionValue"] integerValue];
    NSString *callbackFunc = params[@"callback"];
    
    if (callbackFunc && [callbackFunc length] > 0) {
        [[SolarEngineSDK sharedInstance] updatePostbackConversionValue:conversionValue completionHandler:^(NSError *error) {
            NSLog(@"[lua] updatePostbackConversionValue result: %@", error ?: @"success");
            
            // 回调到 Lua
            int errorCode = error ? (int)error.code : 0;
            [SolarEngineProxy callLuaWithMethod:callbackFunc withIntParam:errorCode];
        }];
    } else {
        [[SolarEngineSDK sharedInstance] updatePostbackConversionValue:conversionValue completionHandler:^(NSError *error) {
            NSLog(@"[lua] updatePostbackConversionValue completed without callback: %@", error ?: @"success");
        }];
    }
}

+ (void)updatePostbackConversionValueWithCoarse:(NSDictionary *)params {
    
    NSInteger fineValue = [params[@"fineValue"] integerValue];
    NSString *coarseValue = params[@"coarseValue"];
    NSString *callbackFunc = params[@"callback"];
    
    if (callbackFunc && [callbackFunc length] > 0) {
        [[SolarEngineSDK sharedInstance] updatePostbackConversionValue:fineValue coarseValue:coarseValue completionHandler:^(NSError *error) {
            NSLog(@"[lua] updatePostbackConversionValueWithCoarse result: %@", error ?: @"success");
            
            // 回调到 Lua
            int errorCode = error ? (int)error.code : 0;
            [SolarEngineProxy callLuaWithMethod:callbackFunc withIntParam:errorCode];
        }];
    } else {
        [[SolarEngineSDK sharedInstance] updatePostbackConversionValue:fineValue coarseValue:coarseValue completionHandler:^(NSError *error) {
            NSLog(@"[lua] updatePostbackConversionValueWithCoarse completed without callback: %@", error ?: @"success");
        }];
    }
}

+ (void)updatePostbackConversionValueWithCoarseAndLock:(NSDictionary *)params {
    
    NSInteger fineValue = [params[@"fineValue"] integerValue];
    NSString *coarseValue = params[@"coarseValue"];
    BOOL lockWindow = [params[@"lockWindow"] boolValue];
    NSString *callbackFunc = params[@"callback"];
    
    if (callbackFunc && [callbackFunc length] > 0) {
        [[SolarEngineSDK sharedInstance] updatePostbackConversionValue:fineValue coarseValue:coarseValue lockWindow:lockWindow completionHandler:^(NSError *error) {
            NSLog(@"[lua] updatePostbackConversionValueWithCoarseAndLock result: %@", error ?: @"success");
            
            // 回调到 Lua
            int errorCode = error ? (int)error.code : 0;
            [SolarEngineProxy callLuaWithMethod:callbackFunc withIntParam:errorCode];
        }];
    } else {
        [[SolarEngineSDK sharedInstance] updatePostbackConversionValue:fineValue coarseValue:coarseValue lockWindow:lockWindow completionHandler:^(NSError *error) {
            NSLog(@"[lua] updatePostbackConversionValueWithCoarseAndLock completed without callback: %@", error ?: @"success");
        }];
    }
}

#pragma mark - RemoteConfig

+ (void)setDefaultConfig:(NSDictionary *)params {
    NSLog(@"[lua] setDefaultConfig called with params: %@", params);
    
    // 检查是否是 JSON 字符串格式
    NSMutableArray *defaultConfig = [[NSMutableArray alloc] init];
        
    for (NSInteger i = 0; i < params.allValues.count; i++) {
        NSString *itemJson = params.allValues[i];
        if ([itemJson isKindOfClass:[NSString class]]) {
            NSData *jsonData = [itemJson dataUsingEncoding:NSUTF8StringEncoding];
            NSDictionary *configItem = [NSJSONSerialization JSONObjectWithData:jsonData options:0 error:nil];
            if (configItem) {
                [defaultConfig addObject:configItem];
            }
        } else if ([itemJson isKindOfClass:[NSDictionary class]]) {
            [defaultConfig addObject:itemJson];
        }
    }
        
    [[SERemoteConfigManager sharedInstance] setDefaultConfig:defaultConfig];
    
}

+ (void)setRemoteConfigEventProperties:(NSDictionary *)params {
    
    NSDictionary *properties = params[@"properties"];
    if (properties && [properties isKindOfClass:[NSDictionary class]]) {
        [[SERemoteConfigManager sharedInstance] setRemoteConfigEventProperties:properties];
        NSLog(@"[lua] setRemoteConfigEventProperties success with properties: %@", properties);
    } else {
        NSLog(@"[lua] setRemoteConfigEventProperties failed: invalid properties parameter");
    }
}

+ (void)setRemoteConfigUserProperties:(NSDictionary *)params {
    
    NSDictionary *properties = params[@"properties"];
    if (properties && [properties isKindOfClass:[NSDictionary class]]) {
        [[SERemoteConfigManager sharedInstance] setRemoteConfigUserProperties:properties];
        NSLog(@"[lua] setRemoteConfigUserProperties success with properties: %@", properties);
    } else {
        NSLog(@"[lua] setRemoteConfigUserProperties failed: invalid properties parameter");
    }
}

+ (NSDictionary *)fastFetchRemoteConfig:(NSDictionary *)params {
    
    NSString *key = params[@"key"];
    if (key && [key isKindOfClass:[NSString class]]) {
        id result = [[SERemoteConfigManager sharedInstance] fastFetchRemoteConfig:key];
        NSLog(@"[lua] fastFetchRemoteConfig result for key '%@': %@", key, result);
        
        if (result) {
            return [NSDictionary dictionaryWithObjectsAndKeys:result, @"value", key, @"key", nil];
        } else {
            return [NSDictionary dictionaryWithObjectsAndKeys:[NSNull null], @"value", key, @"key", nil];
        }
    } else {
        NSLog(@"[lua] fastFetchRemoteConfig failed: invalid key parameter");
        return [NSDictionary dictionaryWithObjectsAndKeys:[NSNull null], @"value", @"", @"key", nil];
    }
}

+ (NSDictionary *)fastFetchAllRemoteConfig:(NSDictionary *)params {
    
    NSDictionary *result = [[SERemoteConfigManager sharedInstance] fastFetchRemoteConfig];
    NSLog(@"[lua] fastFetchAllRemoteConfig result: %@", result);
    
    return result ?: @{};
}

+ (void)asyncFetchRemoteConfig:(NSDictionary *)params {
    NSLog(@"[lua] asyncFetchRemoteConfig called");
    
    NSString *key = params[@"key"];
    NSString *callbackFunc = params[@"callback"];
    
    if (key && [key isKindOfClass:[NSString class]]) {
        [[SERemoteConfigManager sharedInstance] asyncFetchRemoteConfig:key completionHandler:^(id data) {
            NSLog(@"[lua] asyncFetchRemoteConfig result for key '%@': %@", key, data);
            
            if (callbackFunc && [callbackFunc length] > 0) {
                // 创建结果字典
                NSDictionary *resultDict = nil;
                if (data) {
                    resultDict = @{@"value": data, @"key": key};
                } else {
                    resultDict = @{@"value": [NSNull null], @"key": key};
                }
                
                // 转换为 JSON 字符串
                NSString *jsonString = [resultDict toJsonString];
                
                // 回调到 Lua
                [SolarEngineProxy callLuaWithMethod:callbackFunc withStringParam:jsonString];
            }
        }];
    } else {
        NSLog(@"[lua] asyncFetchRemoteConfig failed: invalid key parameter");
        
        if (callbackFunc && [callbackFunc length] > 0) {
            NSDictionary *errorResult = @{@"value": [NSNull null], @"key": @"", @"error": @"Invalid key parameter"};
            NSString *jsonString = [errorResult toJsonString];
            [SolarEngineProxy callLuaWithMethod:callbackFunc withStringParam:jsonString];
        }
    }
}

+ (void)asyncFetchAllRemoteConfig:(NSDictionary *)params {
    NSLog(@"[lua] asyncFetchAllRemoteConfig called");
    
    NSString *callbackFunc = params[@"callback"];
    
    [[SERemoteConfigManager sharedInstance] asyncFetchRemoteConfigWithCompletionHandler:^(NSDictionary *dict) {
        NSLog(@"[lua] asyncFetchAllRemoteConfig result: %@", dict);
        
        if (callbackFunc && [callbackFunc length] > 0) {
            // 转换为 JSON 字符串
            NSDictionary *resultDict = dict ?: @{};
            NSString *jsonString = [resultDict toJsonString];
            
            // 回调到 Lua
            [SolarEngineProxy callLuaWithMethod:callbackFunc withStringParam:jsonString];
        }
    }];
}

@end
