#include "../Common/SolarEngineAPI.h"
#include "../Common/SEJSONObject.h"
#include <stdio.h>
#include <string>
#include <map>
#include <vector>
#include "cocos2d.h"

using namespace solarengine;
USING_NS_CC;

#if CC_TARGET_PLATFORM == CC_PLATFORM_ANDROID

#define SOLARENGINE_JAVA_CLASS "com/solar/engine/SolarEngineCocosAPI"

extern "C"
{
    static string jStringToString(JNIEnv *env, jstring jString)
    {
        if (jString == NULL || env == NULL)
        {
            return "";
        }
        const char *chars = env->GetStringUTFChars(jString, NULL);
        string ret(chars);
        env->ReleaseStringUTFChars(jString, chars);
        return ret;
    }
    static SEJSONObject SEJSONObjectFromJSONObject(JNIEnv *env, jobject object)
    {
        SEJSONObject _object;
        jclass _class = env->GetObjectClass(object);
        jmethodID _keysMethod = env->GetMethodID(_class, "keys", "()Ljava/util/Iterator;");
        jobject _iterator = env->CallObjectMethod(object, _keysMethod);
        jclass _iteratorClass = env->GetObjectClass(_iterator);
        jmethodID _hasNext = env->GetMethodID(_iteratorClass, "hasNext", "()Z");
        jclass _stringClass = env->FindClass("java/lang/String");
        jclass _intClass = env->FindClass("java/lang/Integer");
        jclass _longClass = env->FindClass("java/lang/Long");
        jclass _boolClass = env->FindClass("java/lang/Boolean");
        jclass _jsonArrayClass = env->FindClass("org/json/JSONArray");
        jclass _doubleClass = env->FindClass("java/lang/Double");
        jclass _objectClass = env->FindClass("org/json/JSONObject");
        jmethodID _getDoubleValue = env->GetMethodID(_doubleClass, "doubleValue", "()D");
        jmethodID _getBoolValue = env->GetMethodID(_boolClass, "booleanValue", "()Z");
        jmethodID _getIntValue = env->GetMethodID(_intClass, "intValue", "()I");
        jmethodID _getLongValue = env->GetMethodID(_longClass, "longValue", "()J");

        while (env->CallBooleanMethod(_iterator, _hasNext))
        {
            jmethodID _next = env->GetMethodID(_iteratorClass, "next", "()Ljava/lang/Object;");
            jstring _key = (jstring)env->CallObjectMethod(_iterator, _next);
            jmethodID _get = env->GetMethodID(_class, "get", "(Ljava/lang/String;)Ljava/lang/Object;");
            jobject _value = env->CallObjectMethod(object, _get, _key);
            if (env->IsInstanceOf(_value, _stringClass))
            {
                _object.setString(jStringToString(env, ((jstring)_key)).c_str(),
                                  jStringToString(env, ((jstring)_value)).c_str());
            }
            else if (env->IsInstanceOf(_value, _intClass))
            {
                _object.setNumber(jStringToString(env, ((jstring)_key)).c_str(),
                                  env->CallIntMethod(_value, _getIntValue));
            }
            else if (env->IsInstanceOf(_value, _boolClass))
            {
                _object.setBool(jStringToString(env, ((jstring)_key)).c_str(),
                                env->CallBooleanMethod(_value, _getBoolValue));
            }
            else if (env->IsInstanceOf(_value, _doubleClass))
            {
                _object.setNumber(jStringToString(env, ((jstring)_key)).c_str(),
                                  env->CallDoubleMethod(_value, _getDoubleValue));
            }
            else if (env->IsInstanceOf(_value, _longClass))
            {
                _object.setNumber(jStringToString(env, ((jstring)_key)).c_str(),
                                  env->CallLongMethod(_value, _getLongValue));
            }
            else if (env->IsInstanceOf(_value, _objectClass))
            {
                _object.setJsonObject(jStringToString(env, ((jstring)_key)).c_str(),
                                      SEJSONObjectFromJSONObject(env, _value));
            }
            else
            {
                jmethodID _size = env->GetMethodID(_jsonArrayClass, "length", "()I");
                vector<string> list;
                vector<SEJSONObject> listobjs;
                int i = 0;
                jmethodID _getString = env->GetMethodID(_jsonArrayClass, "getString",
                                                        "(I)Ljava/lang/String;");
                jmethodID _getoptobj = env->GetMethodID(_jsonArrayClass, "optJSONObject",
                                                        "(I)Lorg/json/JSONObject;");
                jmethodID _getobj = env->GetMethodID(_jsonArrayClass, "getJSONObject",
                                                     "(I)Lorg/json/JSONObject;");
                for (; i < env->CallIntMethod(_value, _size); i++)
                {
                    jobject json = (jobject)env->CallObjectMethod(_value, _getoptobj, i);
                    if (json != NULL)
                    {
                        listobjs.push_back(SEJSONObjectFromJSONObject(env, json));
                        env->DeleteLocalRef(json);
                    }
                    else
                    {
                        jstring content = (jstring)env->CallObjectMethod(_value, _getString, i);
                        list.push_back(jStringToString(env, content));
                        env->DeleteLocalRef(content);
                    }
                }
                if (listobjs.size() > 0)
                {
                    _object.setList(jStringToString(env, ((jstring)_key)).c_str(), listobjs);
                }
                else
                {
                    _object.setList(jStringToString(env, ((jstring)_key)).c_str(), list);
                }
            }
            env->DeleteLocalRef(_key);
            env->DeleteLocalRef(_value);
        }
        env->DeleteLocalRef(_class);
        env->DeleteLocalRef(_iterator);
        env->DeleteLocalRef(_iteratorClass);
        env->DeleteLocalRef(_stringClass);
        env->DeleteLocalRef(_intClass);
        env->DeleteLocalRef(_boolClass);
        env->DeleteLocalRef(_longClass);
        env->DeleteLocalRef(_doubleClass);
        env->DeleteLocalRef(_jsonArrayClass);
        return _object;
    }
    static void releaseMethod(JniMethodInfo &method)
    {
        if (method.methodID != NULL)
        {
            method.env->DeleteLocalRef(method.classID);
        }
    }
    static jobject createJavaJsonObject(JNIEnv *env, const solarengine::SEJSONObject *properties)
    {
        jclass classJSONObject = env->FindClass("org/json/JSONObject");
        jmethodID constructMethod = env->GetMethodID(classJSONObject,
                                                     "<init>",
                                                     "(Ljava/lang/String;)V");
        string js = SEJSONObject::toJson(*properties);
        jobject objJSON = env->NewObject(classJSONObject, constructMethod,
                                         env->NewStringUTF(js.c_str()));
        env->DeleteLocalRef(classJSONObject);
        return objJSON;
    }
    static std::string toJsonString(const SEJSONObject &obj)
    {
        return SEJSONObject::toJson(obj);
    }
    static void callJava_setRemoteDefaultConfig(const std::string &jsonArrayString)
    {
        cocos2d::JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo,
                                           SOLARENGINE_JAVA_CLASS,
                                           "setRemoteDefaultConfig",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jArg = methodInfo.env->NewStringUTF(jsonArrayString.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jArg);
            methodInfo.env->DeleteLocalRef(jArg);
            methodInfo.env->DeleteLocalRef(methodInfo.classID);
        }
    }
    static void callJava_setRemoteConfigEventProperties(const std::string &jsonString)
    {
        cocos2d::JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo,
                                           SOLARENGINE_JAVA_CLASS,
                                           "setRemoteConfigEventProperties",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jArg = methodInfo.env->NewStringUTF(jsonString.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jArg);
            methodInfo.env->DeleteLocalRef(jArg);
            methodInfo.env->DeleteLocalRef(methodInfo.classID);
        }
    }
    static void callJava_setRemoteConfigUserProperties(const std::string &jsonString)
    {
        cocos2d::JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo,
                                           SOLARENGINE_JAVA_CLASS,
                                           "setRemoteConfigUserProperties",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jArg = methodInfo.env->NewStringUTF(jsonString.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jArg);
            methodInfo.env->DeleteLocalRef(jArg);
            methodInfo.env->DeleteLocalRef(methodInfo.classID);
        }
    }
    static string callJava_fastFetchRemoteConfigWrap(const std::string &key)
    {
        string result;
        cocos2d::JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo,
                                           SOLARENGINE_JAVA_CLASS,
                                           "fastFetchRemoteConfigWrap",
                                           "(Ljava/lang/String;)Ljava/lang/String;"))
        {
            jstring jKey = methodInfo.env->NewStringUTF(key.c_str());
            jstring jResult = (jstring)methodInfo.env->CallStaticObjectMethod(methodInfo.classID, methodInfo.methodID, jKey);
            if (jResult)
            {
                result = jStringToString(methodInfo.env, jResult);
                methodInfo.env->DeleteLocalRef(jResult);
            }
            methodInfo.env->DeleteLocalRef(jKey);
            releaseMethod(methodInfo);
        }
        return result;
    }
    static void callJava_asyncFetchRemoteConfig(const std::string &key)
    {
        cocos2d::JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo,
                                           SOLARENGINE_JAVA_CLASS,
                                           "asyncFetchRemoteConfig",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jKey = methodInfo.env->NewStringUTF(key.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jKey);
            methodInfo.env->DeleteLocalRef(jKey);
            releaseMethod(methodInfo);
        }
    }
    static SEJSONObject callJava_fastFetchAllRemoteConfigWrap()
    {
        SEJSONObject result;
        cocos2d::JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo,
                                           SOLARENGINE_JAVA_CLASS,
                                           "fastFetchAllRemoteConfigWrap",
                                           "()Lorg/json/JSONObject;"))
        {
            jobject jObj = methodInfo.env->CallStaticObjectMethod(methodInfo.classID, methodInfo.methodID);
            if (jObj)
            {
                result = SEJSONObjectFromJSONObject(methodInfo.env, jObj);
                methodInfo.env->DeleteLocalRef(jObj);
            }
            releaseMethod(methodInfo);
        }
        return result;
    }
    static void callJava_asyncFetchAllRemoteConfig()
    {
        cocos2d::JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo,
                                           SOLARENGINE_JAVA_CLASS,
                                           "asyncFetchAllRemoteConfig",
                                           "()V"))
        {
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID);
            releaseMethod(methodInfo);
        }
    }
    void SolarEngineAPI::preInit(string appKey)
    {
        cocos2d::JniMethodInfo jniGetContext;
        if (!cocos2d::JniHelper::getStaticMethodInfo(jniGetContext,
                                                     "org/cocos2dx/lib/Cocos2dxActivity",
                                                     "getContext",
                                                     "()Landroid/content/Context;"))
        {
            return;
        }
        jobject _context = (jobject)jniGetContext.env->CallStaticObjectMethod(
            jniGetContext.classID, jniGetContext.methodID);
        releaseMethod(jniGetContext);

        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo,
                                           SOLARENGINE_JAVA_CLASS,
                                           "preInit",
                                           "(Landroid/content/Context;Ljava/lang/String;)V"))
        {
            jstring jAppkey = methodInfo.env->NewStringUTF(appKey.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, _context,
                                                 jAppkey);
            methodInfo.env->DeleteLocalRef(_context);
            methodInfo.env->DeleteLocalRef(jAppkey);
            releaseMethod(methodInfo);
        }
    }
    void SolarEngineAPI::init(string appKey, SECConfig config)
    {

        cocos2d::JniMethodInfo jniGetContext;
        if (!cocos2d::JniHelper::getStaticMethodInfo(jniGetContext,
                                                     "org/cocos2dx/lib/Cocos2dxActivity",
                                                     "getContext",
                                                     "()Landroid/content/Context;"))
        {
            return;
        }
        jobject _context = (jobject)jniGetContext.env->CallStaticObjectMethod(jniGetContext.classID, jniGetContext.methodID);
        jniGetContext.env->DeleteLocalRef(jniGetContext.classID);

        cocos2d::JniMethodInfo jniInit;
        if (!cocos2d::JniHelper::getStaticMethodInfo(jniInit,
                                                     SOLARENGINE_JAVA_CLASS,
                                                     "init",
                                                     "(Landroid/content/Context;Ljava/lang/String;Ljava/lang/String;)V"))
        {
            return;
        }
        string configJson = se_configToJsonString(config);
        jstring jConfigJson = jniInit.env->NewStringUTF(configJson.c_str());
        jstring jAppKey = jniInit.env->NewStringUTF(appKey.c_str());
        jniInit.env->CallStaticVoidMethod(jniInit.classID, jniInit.methodID, _context, jAppKey,jConfigJson);
        jniInit.env->DeleteLocalRef(jConfigJson);
        jniInit.env->DeleteLocalRef(jAppKey);
        jniInit.env->DeleteLocalRef(_context);
    }

    string SolarEngineAPI::se_configToJsonString(SECConfig config)
    {
        string jsonStr = "{";
        jsonStr += "\"enableLog\":" + string(config.enableLog ? "true" : "false") + ",";
        jsonStr += "\"isDebugModel\":" + string(config.isDebugModel ? "true" : "false") + ",";
        jsonStr += "\"isGDPRArea\":" + string(config.isGDPRArea ? "true" : "false") + ",";
        jsonStr += "\"setCoppaEnabled\":" + string(config.setCoppaEnabled ? "true" : "false") + ",";
        jsonStr += "\"setKidsAppEnabled\":" + string(config.setKidsAppEnabled ? "true" : "false") + ",";
        jsonStr += "\"enable2GReporting\":" + string(config.enable2GReporting ? "true" : "false") + ",";
        jsonStr += "\"enableDeferredDeeplink\":" +string(config.enableDeferredDeeplink ? "true" : "false")+ ",";

        jsonStr += "\"adPersonalizationEnabled\":" +string(config.adPersonalizationEnabled ? "true" : "false")+ ",";
        jsonStr += "\"adUserDataEnabled\":" +string(config.adUserDataEnabled ? "true" : "false")+ ",";
        jsonStr += "\"supportMultiProcess\":" +string(config.supportMultiProcess ? "true" : "false")+ ",";
        jsonStr += "\"isOAIDEnabled\":" +string(config.isOAIDEnabled ? "true" : "false")+ ",";
        jsonStr += "\"isImeiEnabled\":" +string(config.isImeiEnabled ? "true" : "false")+ ",";
        jsonStr += "\"isAndroidIDEnabled\":" +string(config.isAndroidIDEnabled ? "true" : "false")+ ",";
        jsonStr += "\"enableIPV6Address\":" +string(config.enableIPV6Address ? "true" : "false")+ ",";

        jsonStr += "\"sub_lib_version\":" +SE_COCOS_LIB_VERSION;

        if (config.remoteConfig.enable)
        {
            jsonStr += ",\"remoteConfig\":{";
            jsonStr += "\"enable\":" + string(config.remoteConfig.enable ? "true" : "false") + ",";
            jsonStr += "\"mergeType\":" + to_string(config.remoteConfig.mergeType);
            if (!config.remoteConfig.customIDEventProperties.propertiesMap.empty())
            {
                jsonStr += ",\"customIDEventProperties\":" + SEJSONObject::toJson(config.remoteConfig.customIDEventProperties);
            }
            if (!config.remoteConfig.customIDUserProperties.propertiesMap.empty())
            {
                jsonStr += ",\"customIDUserProperties\":" + SEJSONObject::toJson(config.remoteConfig.customIDUserProperties);
            }
            jsonStr += "}";
        }

        if (config.customDomain.enable)
        {
            jsonStr += ",\"customDomain\":{";
            jsonStr += "\"enable\":" + string(config.customDomain.enable ? "true" : "false") + ",";
            jsonStr += "\"receiverDomain\":\"" + config.customDomain.receiverDomain + "\",";
            jsonStr += "\"ruleDomain\":\"" + config.customDomain.ruleDomain + "\",";
            jsonStr += "\"receiverTcpHost\":\"" + config.customDomain.receiverTcpHost + "\",";
            jsonStr += "\"ruleTcpHost\":\"" + config.customDomain.ruleTcpHost + "\",";
            jsonStr += "\"gatewayTcpHost\":\"" + config.customDomain.gatewayTcpHost + "\"";
            jsonStr += "}";
        }
        jsonStr += "}";
        return jsonStr;
    }

    void SolarEngineAPI::setGDPRArea(bool isGDPRArea)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "setGDPRArea", "(Z)V"))
        {
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, isGDPRArea);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::track(SECustomEvent *event)
    {
        if (!event){return;}
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackCustomEvent",
                                           "(Ljava/lang/String;Lorg/json/JSONObject;Lorg/json/JSONObject;)V"))
        {
            jstring jEventName = methodInfo.env->NewStringUTF(event->eventName.c_str());
            jobject jCustomProps = createJavaJsonObject(methodInfo.env, &event->customProperties);
            jobject jPresetProps = createJavaJsonObject(methodInfo.env, &event->presetProperties);

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID,jEventName, jCustomProps, jPresetProps);

            methodInfo.env->DeleteLocalRef(jEventName);
            methodInfo.env->DeleteLocalRef(jCustomProps);
            methodInfo.env->DeleteLocalRef(jPresetProps);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::track(SERegisterEvent *event)
    {
        if (!event){return;}
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackRegisterEvent",
                                           "(Ljava/lang/String;Ljava/lang/String;Lorg/json/JSONObject;)V"))
        {
            jstring jRegisterType = methodInfo.env->NewStringUTF(event->registerType.c_str());
            jstring jRegisterStatus = methodInfo.env->NewStringUTF(event->registerStatus.c_str());
            jobject jCustomProps = createJavaJsonObject(methodInfo.env, &event->customProperties);

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID,jRegisterType, jRegisterStatus, jCustomProps);

            methodInfo.env->DeleteLocalRef(jRegisterType);
            methodInfo.env->DeleteLocalRef(jRegisterStatus);
            methodInfo.env->DeleteLocalRef(jCustomProps);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::track(SEPurchaseEvent *event)
    {
        if (!event){return;}

        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackIAPEvent",
                                           "(Ljava/lang/String;Ljava/lang/String;IDLjava/lang/String;Ljava/lang/String;Ljava/lang/String;ILjava/lang/String;Lorg/json/JSONObject;)V"))
        {
            jstring jProductID = methodInfo.env->NewStringUTF(event->productId.c_str());
            jstring jProductName = methodInfo.env->NewStringUTF(event->productName.c_str());
            jstring jOrderId = methodInfo.env->NewStringUTF(event->orderId.c_str());
            jstring jCurrencyType = methodInfo.env->NewStringUTF(event->currencyType.c_str());
            jstring jPayType = methodInfo.env->NewStringUTF(event->payType.c_str());
            jstring jFailReason = methodInfo.env->NewStringUTF(event->failReason.c_str());
            jobject jCustomProps = createJavaJsonObject(methodInfo.env, &event->customProperties);

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jProductID, jProductName, event->productNum,
                                                 event->payAmount, jOrderId, jCurrencyType, jPayType, event->payStatus, jFailReason, jCustomProps);

            methodInfo.env->DeleteLocalRef(jProductID);
            methodInfo.env->DeleteLocalRef(jProductName);
            methodInfo.env->DeleteLocalRef(jOrderId);
            methodInfo.env->DeleteLocalRef(jCurrencyType);
            methodInfo.env->DeleteLocalRef(jPayType);
            methodInfo.env->DeleteLocalRef(jFailReason);
            methodInfo.env->DeleteLocalRef(jCustomProps);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::track(SEAdImpressionEvent *event)
    {
        if (!event){return;}

        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackAdImpressionEvent",
                                           "(ILjava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;DLjava/lang/String;ZLorg/json/JSONObject;)V"))
        {
            jstring jAdNetworkPlatform = methodInfo.env->NewStringUTF(event->adNetworkPlatform.c_str());
            jstring jAdNetworkAppID = methodInfo.env->NewStringUTF(event->adNetworkAppID.c_str());
            jstring jAdNetworkPlacementID = methodInfo.env->NewStringUTF(
                event->adNetworkPlacementID.c_str());
            jstring jCurrency = methodInfo.env->NewStringUTF(event->currency.c_str());
            jstring jMediationPlatform = methodInfo.env->NewStringUTF(event->mediationPlatform.c_str());
            jobject jCustomProps = createJavaJsonObject(methodInfo.env, &event->customProperties);

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID,
                                                 event->adType, jAdNetworkPlatform, jAdNetworkAppID,
                                                 jAdNetworkPlacementID, jCurrency, event->ecpm,
                                                 jMediationPlatform, event->rendered, jCustomProps);

            methodInfo.env->DeleteLocalRef(jAdNetworkPlatform);
            methodInfo.env->DeleteLocalRef(jAdNetworkAppID);
            methodInfo.env->DeleteLocalRef(jAdNetworkPlacementID);
            methodInfo.env->DeleteLocalRef(jCurrency);
            methodInfo.env->DeleteLocalRef(jMediationPlatform);
            methodInfo.env->DeleteLocalRef(jCustomProps);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::track(SEAdClickEvent *event)
    {
        if (!event){return;}

        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackAdClickEvent",
                                           "(ILjava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Lorg/json/JSONObject;)V"))
        {
            jstring jAdNetworkPlatform = methodInfo.env->NewStringUTF(event->adNetworkPlatform.c_str());
            jstring jAdNetworkPlacementID = methodInfo.env->NewStringUTF(
                event->adNetworkPlacementID.c_str());
            jstring jMediationPlatform = methodInfo.env->NewStringUTF(event->mediationPlatform.c_str());
            jstring jAdNetworkAppID = methodInfo.env->NewStringUTF(event->adNetworkPlacementID.c_str());
            jobject jCustomProps = createJavaJsonObject(methodInfo.env, &event->customProperties);

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID,
                                                 event->adType, jAdNetworkPlatform,
                                                 jAdNetworkPlacementID,
                                                 jMediationPlatform, jAdNetworkAppID, jCustomProps);

            methodInfo.env->DeleteLocalRef(jAdNetworkPlatform);
            methodInfo.env->DeleteLocalRef(jAdNetworkPlacementID);
            methodInfo.env->DeleteLocalRef(jMediationPlatform);
            methodInfo.env->DeleteLocalRef(jAdNetworkAppID);
            methodInfo.env->DeleteLocalRef(jCustomProps);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::track(SEAppAttrEvent *event)
    {
        if (!event){return;}

        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackAppAttrEvent",
                                           "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Lorg/json/JSONObject;)V"))
        {
            jstring jAdNetwork = methodInfo.env->NewStringUTF(event->adNetwork.c_str());
            jstring jSubChannel = methodInfo.env->NewStringUTF(event->subChannel.c_str());
            jstring jAdAccountID = methodInfo.env->NewStringUTF(event->adAccountId.c_str());
            jstring jAdAccountName = methodInfo.env->NewStringUTF(event->adAccountName.c_str());
            jstring jAdCampaignID = methodInfo.env->NewStringUTF(event->adCampaignId.c_str());
            jstring jAdCampaignName = methodInfo.env->NewStringUTF(event->adCampaignName.c_str());
            jstring jAdOfferID = methodInfo.env->NewStringUTF(event->adOfferId.c_str());
            jstring jAdOfferName = methodInfo.env->NewStringUTF(event->adOfferName.c_str());
            jstring jAdCreativeID = methodInfo.env->NewStringUTF(event->adCreativeId.c_str());
            jstring jAdCreativeName = methodInfo.env->NewStringUTF(event->adCreativeName.c_str());
            jstring jAttributionPlatform = methodInfo.env->NewStringUTF(
                event->attributionPlatform.c_str());
            jobject jCustomProps = createJavaJsonObject(methodInfo.env, &event->customProperties);

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID,
                                                 jAdNetwork, jSubChannel, jAdAccountID, jAdAccountName,
                                                 jAdCampaignID, jAdCampaignName, jAdOfferID,
                                                 jAdOfferName,
                                                 jAdCreativeID, jAdCreativeName, jAttributionPlatform,
                                                 jCustomProps);

            methodInfo.env->DeleteLocalRef(jAdNetwork);
            methodInfo.env->DeleteLocalRef(jSubChannel);
            methodInfo.env->DeleteLocalRef(jAdAccountID);
            methodInfo.env->DeleteLocalRef(jAdAccountName);
            methodInfo.env->DeleteLocalRef(jAdCampaignID);
            methodInfo.env->DeleteLocalRef(jAdCampaignName);
            methodInfo.env->DeleteLocalRef(jAdOfferID);
            methodInfo.env->DeleteLocalRef(jAdOfferName);
            methodInfo.env->DeleteLocalRef(jAdCreativeID);
            methodInfo.env->DeleteLocalRef(jAdCreativeName);
            methodInfo.env->DeleteLocalRef(jAttributionPlatform);
            methodInfo.env->DeleteLocalRef(jCustomProps);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::track(SELoginEvent *event)
    {
        if (!event){return;}

        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackLoginEvent",
                                           "(Ljava/lang/String;Ljava/lang/String;Lorg/json/JSONObject;)V"))
        {
            jstring jLoginType = methodInfo.env->NewStringUTF(event->loginType.c_str());
            jstring jLoginStatus = methodInfo.env->NewStringUTF(event->loginStatus.c_str());
            jobject jCustomProps = createJavaJsonObject(methodInfo.env, &event->customProperties);

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID,jLoginType, jLoginStatus, jCustomProps);

            methodInfo.env->DeleteLocalRef(jLoginType);
            methodInfo.env->DeleteLocalRef(jLoginStatus);
            methodInfo.env->DeleteLocalRef(jCustomProps);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::track(SEOrderEvent *event)
    {
        if (!event){return;}

        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackOrderEvent",
                                           "(Ljava/lang/String;DLjava/lang/String;Ljava/lang/String;Ljava/lang/String;Lorg/json/JSONObject;)V"))
        {
            jstring jOrderID = methodInfo.env->NewStringUTF(event->orderId.c_str());
            jstring jCurrencyType = methodInfo.env->NewStringUTF(event->currencyType.c_str());
            jstring jPayType = methodInfo.env->NewStringUTF(event->payType.c_str());
            jstring jStatus = methodInfo.env->NewStringUTF(event->status.c_str());
            jobject jCustomProps = createJavaJsonObject(methodInfo.env, &event->customProperties);

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID,
                                                 jOrderID, event->payAmount, jCurrencyType,
                                                 jPayType, jStatus, jCustomProps);

            methodInfo.env->DeleteLocalRef(jOrderID);
            methodInfo.env->DeleteLocalRef(jCurrencyType);
            methodInfo.env->DeleteLocalRef(jPayType);
            methodInfo.env->DeleteLocalRef(jStatus);
            methodInfo.env->DeleteLocalRef(jCustomProps);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::trackFirstEvent(string eventName, string firstCheckId,
                                         const SEJSONObject &properties)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackFirstEvent",
                                           "(Ljava/lang/String;Ljava/lang/String;Lorg/json/JSONObject;)V"))
        {
            jstring jEventName = methodInfo.env->NewStringUTF(eventName.c_str());
            jstring jFirstCheckId = methodInfo.env->NewStringUTF(firstCheckId.c_str());
            jobject jProperties = createJavaJsonObject(methodInfo.env, &properties);

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID,jEventName, jFirstCheckId, jProperties);

            methodInfo.env->DeleteLocalRef(jEventName);
            methodInfo.env->DeleteLocalRef(jFirstCheckId);
            methodInfo.env->DeleteLocalRef(jProperties);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::eventStart(string eventName)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "eventStart",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jEventName = methodInfo.env->NewStringUTF(eventName.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jEventName);
            methodInfo.env->DeleteLocalRef(jEventName);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::eventFinish(string eventName, const SEJSONObject &properties)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "eventFinish",
                                           "(Ljava/lang/String;Lorg/json/JSONObject;)V"))
        {
            jstring jEventName = methodInfo.env->NewStringUTF(eventName.c_str());
            jobject jProperties = createJavaJsonObject(methodInfo.env, &properties);
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jEventName,jProperties);
            methodInfo.env->DeleteLocalRef(jEventName);
            methodInfo.env->DeleteLocalRef(jProperties);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::setPresetEvent(SECPresetEventType eventType, const SEJSONObject &properties)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "setPresetEvent",
                                           "(ILorg/json/JSONObject;)V"))
        {
            jobject jProperties = createJavaJsonObject(methodInfo.env, &properties);
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, eventType,jProperties);
            methodInfo.env->DeleteLocalRef(jProperties);
            releaseMethod(methodInfo);
        }
    }

    SEPresetProperties SolarEngineAPI::getPresetProperties()
    {
        SEPresetProperties preset;
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "getPresetProperties",
                                           "()Lorg/json/JSONObject;"))
        {
            jobject jResult = methodInfo.env->CallStaticObjectMethod(methodInfo.classID,
                                                                     methodInfo.methodID);
            if (jResult)
            {
                SEJSONObject result = SEJSONObjectFromJSONObject(methodInfo.env, jResult);
                methodInfo.env->DeleteLocalRef(jResult);

                auto getString = [&](const char *key) -> std::string
                {
                    auto it = result.propertiesMap.find(key);
                    if (it == result.propertiesMap.end()){return std::string();}
                    std::string out;
                    SEJSONObject::ValueNode::toStr(it->second, &out);
                    if (!out.empty() && out.front() == '"' && out.back() == '"')
                        out = out.substr(1, out.size() - 2);
                    return out;
                };
                auto getInt = [&](const char *key) -> int
                {
                    auto it = result.propertiesMap.find(key);
                    if (it == result.propertiesMap.end()){return 0;}
                    std::string out;
                    SEJSONObject::ValueNode::toStr(it->second, &out);
                    try
                    {
                        return (int)std::stoll(out);
                    }
                    catch (...)
                    {
                        return 0;
                    }
                };
                auto getDouble = [&](const char *key) -> double
                {
                    auto it = result.propertiesMap.find(key);
                    if (it == result.propertiesMap.end()){return 0.0;}
                    std::string out;
                    SEJSONObject::ValueNode::toStr(it->second, &out);
                    try
                    {
                        return std::stod(out);
                    }
                    catch (...)
                    {
                        return 0.0;
                    }
                };

                preset.appkey = getString("_appkey");
                preset.distinctId = getString("_distinct_id");
                preset.accountId = getString("_account_id");
                preset.visitorId = getString("_visitor_id");
                preset.sessionId = getString("_session_id");
                preset.uuid = getString("_uuid");
                preset.imei = getString("_imei");
                preset.imei2 = getString("_imei2");
                preset.gaid = getString("_gaid");
                preset.oaid = getString("_oaid");
                preset.androidId = getString("_android_id");
                preset.ua = getString("_ua");
                preset.language = getString("_language");
                preset.timeZone = getString("_time_zone");
                preset.manufacturer = getString("_manufacturer");
                preset.platform = getString("_platform");
                preset.osVersion = getString("_os_version");
                preset.screenHeight = getInt("_screen_height");
                preset.screenWidth = getInt("_screen_width");
                preset.density = getDouble("_density");
                preset.deviceModel = getString("_device_model");
                preset.deviceType = getInt("_device_type");
                preset.appVersion = getString("_app_version");
                preset.appVersionCode = getInt("_app_version_code");
                preset.packageName = getString("_package_name");
                preset.appName = getString("_app_name");
                preset.channel = getString("_channel");
                preset.lib = getString("_lib");
                preset.libVersion = getString("_lib_version");
            }
            releaseMethod(methodInfo);
        }
        return preset;
    }

    AttributionData SolarEngineAPI::getAttributionData()
    {
        SEJSONObject dataObj;
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "getAttributionData",
                                           "()Lorg/json/JSONObject;"))
        {
            jobject jResult = methodInfo.env->CallStaticObjectMethod(methodInfo.classID,methodInfo.methodID);
            if (jResult)
            {
                dataObj = SEJSONObjectFromJSONObject(methodInfo.env, jResult);
                methodInfo.env->DeleteLocalRef(jResult);
            }
            releaseMethod(methodInfo);
        }

        auto getString = [&](const char *key) -> std::string
        {
            auto it = dataObj.propertiesMap.find(key);
            if (it == dataObj.propertiesMap.end()){return std::string();}
            std::string out;
            SEJSONObject::ValueNode::toStr(it->second, &out);
            if (!out.empty() && out.front() == '"' && out.back() == '"')
                out = out.substr(1, out.size() - 2);
            return out;
        };

        AttributionData attrData;
        if (dataObj.propertiesMap.empty())
        {
            attrData.code = -1;
        }
        else
        {
            attrData.code = 0;
            attrData.account_id = getString("account_id");
            attrData.ad_type = getString("ad_type");
            attrData.adcreative_id = getString("adcreative_id");
            attrData.adcreative_name = getString("adcreative_name");
            attrData.adcreative_type = getString("adcreative_type");
            attrData.adgroup_id = getString("adgroup_id");
            attrData.adgroup_name = getString("adgroup_name");
            attrData.adplan_id = getString("adplan_id");
            attrData.adplan_name = getString("adplan_name");
            attrData.attribution_time = getString("attribution_time");
            attrData.attribution_touch_type = getString("attribution_touch_type");
            attrData.attribution_type = getString("attribution_type");
            attrData.callback_id = getString("callback_id");
            attrData.channel_id = getString("channel_id");
            attrData.channel_name = getString("channel_name");
            attrData.click_id = getString("click_id");
            attrData.conversion_id = getString("conversion_id");
            attrData.impression_id = getString("impression_id");
            attrData.install_time = getString("install_time");
            attrData.placement_id = getString("placement_id");
            attrData.report_time = getString("report_time");
            attrData.request_id = getString("request_id");
            attrData.ry_touchpoint_ts = getString("ry_touchpoint_ts");
            attrData.site_id = getString("site_id");
            attrData.site_name = getString("site_name");
            attrData.turl_campaign_id = getString("turl_campaign_id");
            attrData.turl_campaign_name = getString("turl_campaign_name");
            attrData.turl_id = getString("turl_id");

            attrData.origin_data_str = SEJSONObject::toJson(dataObj);
        }

        return attrData;
    }

    void SolarEngineAPI::appDeeplinkOpenURL(string url)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "appDeeplinkOpenURL",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jUrl = methodInfo.env->NewStringUTF(url.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jUrl);
            methodInfo.env->DeleteLocalRef(jUrl);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::setGaid(string gaid) {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "setGaid",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jGaid = methodInfo.env->NewStringUTF(gaid.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jGaid);
            methodInfo.env->DeleteLocalRef(jGaid);
            releaseMethod(methodInfo);
        }
    }
    void SolarEngineAPI::setChannel(string channel) {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "setChannel",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jChannel = methodInfo.env->NewStringUTF(channel.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jChannel);
            methodInfo.env->DeleteLocalRef(jChannel);
            releaseMethod(methodInfo);
        }
    }
    void SolarEngineAPI::requestTrackingAuthorization(std::function<void(int status)> completionHandler) {
        // only ios
    }

// 访客ID管理
    void SolarEngineAPI::setVisitorID(string visitorId)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "setVisitorID",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jVisitorId = methodInfo.env->NewStringUTF(visitorId.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jVisitorId);
            methodInfo.env->DeleteLocalRef(jVisitorId);
            releaseMethod(methodInfo);
        }
    }

    string SolarEngineAPI::getVisitorID()
    {
        string result;
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "getVisitorID",
                                           "()Ljava/lang/String;"))
        {
            jstring jResult = (jstring)methodInfo.env->CallStaticObjectMethod(methodInfo.classID,
                                                                              methodInfo.methodID);
            if (jResult)
            {
                result = jStringToString(methodInfo.env, jResult);
                methodInfo.env->DeleteLocalRef(jResult);
            }
            releaseMethod(methodInfo);
        }
        return result;
    }

    void SolarEngineAPI::reportEventImmediately()
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "reportEventImmediately",
                                           "()V"))
        {
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::loginWithAccountID(string accountId)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "loginWithAccountID",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jAccountId = methodInfo.env->NewStringUTF(accountId.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jAccountId);
            methodInfo.env->DeleteLocalRef(jAccountId);
            releaseMethod(methodInfo);
        }
    }

    string SolarEngineAPI::getAccountID()
    {
        string result;
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "getAccountID",
                                           "()Ljava/lang/String;"))
        {
            jstring jResult = (jstring)methodInfo.env->CallStaticObjectMethod(methodInfo.classID,
                                                                              methodInfo.methodID);
            if (jResult)
            {
                result = jStringToString(methodInfo.env, jResult);
                methodInfo.env->DeleteLocalRef(jResult);
            }
            releaseMethod(methodInfo);
        }
        return result;
    }

    void SolarEngineAPI::logout()
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "logout", "()V"))
        {
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID);
            releaseMethod(methodInfo);
        }
    }

    string SolarEngineAPI::getDistinctId()
    {
        string result;
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "getDistinctId",
                                           "()Ljava/lang/String;"))
        {
            jstring jResult = (jstring)methodInfo.env->CallStaticObjectMethod(methodInfo.classID,
                                                                              methodInfo.methodID);
            if (jResult)
            {
                result = jStringToString(methodInfo.env, jResult);
                methodInfo.env->DeleteLocalRef(jResult);
            }
            releaseMethod(methodInfo);
        }
        return result;
    }

    // 公共事件属性
    void SolarEngineAPI::setSuperProperties(const SEJSONObject &properties)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "setSuperProperties",
                                           "(Lorg/json/JSONObject;)V"))
        {
            jobject jProperties = createJavaJsonObject(methodInfo.env, &properties);
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jProperties);
            methodInfo.env->DeleteLocalRef(jProperties);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::unsetSuperProperty(string key)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "unsetSuperProperty",
                                           "(Ljava/lang/String;)V"))
        {
            jstring jKey = methodInfo.env->NewStringUTF(key.c_str());
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jKey);
            methodInfo.env->DeleteLocalRef(jKey);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::clearSuperProperties()
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "clearSuperProperties",
                                           "()V"))
        {
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID);
            releaseMethod(methodInfo);
        }
    }

    // 用户属性管理
    void SolarEngineAPI::userInit(const SEJSONObject &properties)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "userInit",
                                           "(Lorg/json/JSONObject;)V"))
        {
            jobject jProperties = createJavaJsonObject(methodInfo.env, &properties);
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jProperties);
            methodInfo.env->DeleteLocalRef(jProperties);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::userUpdate(const SEJSONObject &properties)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "userUpdate",
                                           "(Lorg/json/JSONObject;)V"))
        {
            jobject jProperties = createJavaJsonObject(methodInfo.env, &properties);
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jProperties);
            methodInfo.env->DeleteLocalRef(jProperties);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::userAdd(const SEJSONObject &properties)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "userAdd",
                                           "(Lorg/json/JSONObject;)V"))
        {
            jobject jProperties = createJavaJsonObject(methodInfo.env, &properties);
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jProperties);
            methodInfo.env->DeleteLocalRef(jProperties);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::userUnset(const vector<string> &keys)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "userUnset",
                                           "([Ljava/lang/String;)V"))
        {
            jobjectArray jKeys = methodInfo.env->NewObjectArray(keys.size(),
                                                                methodInfo.env->FindClass(
                                                                    "java/lang/String"),
                                                                NULL);
            for (size_t i = 0; i < keys.size(); i++)
            {
                jstring jKey = methodInfo.env->NewStringUTF(keys[i].c_str());
                methodInfo.env->SetObjectArrayElement(jKeys, i, jKey);
                methodInfo.env->DeleteLocalRef(jKey);
            }
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jKeys);
            methodInfo.env->DeleteLocalRef(jKeys);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::userAppend(const SEJSONObject &properties)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "userAppend",
                                           "(Lorg/json/JSONObject;)V"))
        {
            jobject jProperties = createJavaJsonObject(methodInfo.env, &properties);
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jProperties);
            methodInfo.env->DeleteLocalRef(jProperties);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::userDelete(SECDeleteType deleteType)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "userDelete", "(I)V"))
        {
            int type = -1;
            if (deleteType == byAccountId) {
                type = 0;
            } else if (deleteType == byVisitorId) {
                type = 1;
            }

            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, type);
            releaseMethod(methodInfo);
        }
    }

    void SolarEngineAPI::trackAppReEngagement(const SEJSONObject &customProperties)
    {
        JniMethodInfo methodInfo;
        if (JniHelper::getStaticMethodInfo(methodInfo, SOLARENGINE_JAVA_CLASS, "trackAppReEngagement",
                                           "(Lorg/json/JSONObject;)V"))
        {
            jobject jCustomProps = createJavaJsonObject(methodInfo.env, &customProperties);
            methodInfo.env->CallStaticVoidMethod(methodInfo.classID, methodInfo.methodID, jCustomProps);
            methodInfo.env->DeleteLocalRef(jCustomProps);
            releaseMethod(methodInfo);
        }
    }

    std::function<void(int)> SolarEngineAPI::se_initCallback = nullptr;
    std::function<void(const AttributionData &)> SolarEngineAPI::se_attributionDataCallback = nullptr;
    std::function<void(const DeferredDeeplinkData &)> SolarEngineAPI::se_deferredDeeplinkCallback = nullptr;
    std::function<void(const DeeplinkData &)> SolarEngineAPI::se_deeplinkCallback = nullptr;

    void SolarEngineAPI::se_onInitCallback(int resultCode)
    {
        if (se_initCallback)
        {
            se_initCallback(resultCode);
        }
    }
    void SolarEngineAPI::se_onAttributionDataCallback(const AttributionData &data)
    {
        if (se_attributionDataCallback)
        {
            se_attributionDataCallback(data);
        }
    }
    void SolarEngineAPI::se_onDeferredDeeplinkCallback(const DeferredDeeplinkData &data)
    {
        if (se_deferredDeeplinkCallback)
        {
            se_deferredDeeplinkCallback(data);
        }
    }
    void SolarEngineAPI::se_onDeeplinkCallback(const DeeplinkData &data)
    {
        if (se_deeplinkCallback)
        {
            se_deeplinkCallback(data);
        }
    }

    void SolarEngineAPI::setInitCallback(std::function<void(int)> callback)
    {
        se_initCallback = callback;
    }
    void SolarEngineAPI::setAttributionDataCallback(std::function<void(const AttributionData &)> callback)
    {
        se_attributionDataCallback = callback;
    }
    void SolarEngineAPI::setDeferredDeeplinkCallback(
        std::function<void(const DeferredDeeplinkData &)> callback)
    {
        se_deferredDeeplinkCallback = callback;
    }
    void SolarEngineAPI::setDeeplinkCallback(std::function<void(const DeeplinkData &)> callback)
    {
        se_deeplinkCallback = callback;
    }

    void SolarEngineAPI::setDefaultRemoteConfig(const std::vector<SECRemoteConfigItem> &defaultConfig)
    {
        auto jsonEscape = [](const std::string &in) -> std::string
        {
            std::string out;
            out.reserve(in.size() + 8);
            for (char c : in)
            {
                switch (c)
                {
                case '\\':
                    out += "\\\\";
                    break;
                case '"':
                    out += "\\\"";
                    break;
                case '\n':
                    out += "\\n";
                    break;
                case '\r':
                    out += "\\r";
                    break;
                case '\t':
                    out += "\\t";
                    break;
                default:
                    out += c;
                    break;
                }
            }
            return out;
        };
        auto trim = [](std::string s) -> std::string
        {
            size_t i = 0, j = s.size();
            while (i < j && isspace((unsigned char)s[i]))
                ++i;
            while (j > i && isspace((unsigned char)s[j - 1]))
                --j;
            return s.substr(i, j - i);
        };
        auto looksLikeJson = [&](const std::string &s) -> bool
        {
            std::string t = trim(s);
            if (t.empty())
                return false;
            return (t.front() == '{' && t.back() == '}') || (t.front() == '[' && t.back() == ']');
        };

        std::string json = "[";
        for (size_t i = 0; i < defaultConfig.size(); ++i)
        {
            const auto &item = defaultConfig[i];
            json += "{";
            json += "\"name\":\"" + jsonEscape(item.name) + "\",";
            json += "\"type\":" + std::to_string(item.type) + ",";

            if (item.type == 1)
            {
                // string
                json += "\"value\":\"" + jsonEscape(item.value) + "\"";
            }
            else if (item.type == 2)
            {
                // integer
                json += "\"value\":" + (item.value.empty() ? std::string("0") : item.value);
            }
            else if (item.type == 3)
            {
                // boolean
                std::string v = item.value;
                for (auto &ch : v)
                    ch = (char)tolower((unsigned char)ch);
                bool b = (v == "true" || v == "1" || v == "yes");
                json += std::string("\"value\":") + (b ? "true" : "false");
            }
            else if (item.type == 4)
            {
                // json (object/array). If valid JSON, embed raw; otherwise, fall back to quoted string
                if (looksLikeJson(item.value))
                {
                    json += "\"value\":" + trim(item.value);
                }
                else
                {
                    json += "\"value\":\"" + jsonEscape(item.value) + "\"";
                }
            }
            else
            {
                json += "\"value\":\"" + jsonEscape(item.value) + "\"";
            }

            json += "}";
            if (i + 1 < defaultConfig.size())
                json += ",";
        }
        json += "]";
        callJava_setRemoteDefaultConfig(json);
    }

    void SolarEngineAPI::setRemoteConfigEventProperties(const SEJSONObject &properties)
    {
        callJava_setRemoteConfigEventProperties(SEJSONObject::toJson(properties));
    }

    void SolarEngineAPI::setRemoteConfigUserProperties(const SEJSONObject &properties)
    {
        callJava_setRemoteConfigUserProperties(SEJSONObject::toJson(properties));
    }

    string SolarEngineAPI::fastFetchRemoteConfig(const std::string &key)
    {
        return callJava_fastFetchRemoteConfigWrap(key);
    }

    // 异步：将结果回调到 C++ 的使用方，由外部先设置回调（可复用已有回调机制或用户自行传lambda）
    static std::function<void(const string &)> g_remoteConfigCallback;

    void SolarEngineAPI::asyncFetchRemoteConfig(const std::string &key, std::function<void(const string &)> completion)
    {
        g_remoteConfigCallback = completion;
        callJava_asyncFetchRemoteConfig(key);
    }

    // 远程配置：同步获取全部参数
    SEJSONObject SolarEngineAPI::fastFetchAllRemoteConfig()
    {
        return callJava_fastFetchAllRemoteConfigWrap();
    }

    static std::function<void(const SEJSONObject &)> g_remoteConfigAllCallback;

    void SolarEngineAPI::asyncFetchAllRemoteConfig(std::function<void(const SEJSONObject &)> completion)
    {
        g_remoteConfigAllCallback = completion;
        callJava_asyncFetchAllRemoteConfig();
    }

    // Java -> C++ 回调
    JNIEXPORT void JNICALL Java_com_solar_engine_SolarEngineCocosAPI_nativeOnRemoteConfigResult(JNIEnv *env, jclass clazz, jstring jResult)
    {
        if (!g_remoteConfigCallback)
            return;
        string result;
        if (jResult)
        {
            result = jStringToString(env, jResult);
        }
        // 切回主线程回调
        cocos2d::Director::getInstance()->getScheduler()->performFunctionInCocosThread([result]()
                                                                                       {
            if (g_remoteConfigCallback) g_remoteConfigCallback(result); });
    }

    // Java -> C++ 回调（全部）
    JNIEXPORT void JNICALL Java_com_solar_engine_SolarEngineCocosAPI_nativeOnRemoteConfigAllResult(JNIEnv *env, jclass clazz, jobject jResult)
    {
        if (!g_remoteConfigAllCallback)
            return;
        SEJSONObject result;
        if (jResult)
        {
            result = SEJSONObjectFromJSONObject(env, jResult);
        }
        cocos2d::Director::getInstance()->getScheduler()->performFunctionInCocosThread([result]()
                                                                                       {
            if (g_remoteConfigAllCallback) g_remoteConfigAllCallback(result); });
    }
}

extern "C" JNIEXPORT void JNICALL
Java_com_solar_engine_SolarEngineCocosAPI_nativeOnInitCompleted(JNIEnv *env, jclass clazz,jint resultCode)
{
    // 切回 Cocos 主线程再分发（仅当监听已设置时才会真正回调）
    cocos2d::Director::getInstance()->getScheduler()->performFunctionInCocosThread(
        [resultCode]()
        { solarengine::SolarEngineAPI::se_onInitCallback((int)resultCode); });
}

extern "C" JNIEXPORT void JNICALL
Java_com_solar_engine_SolarEngineCocosAPI_nativeOnAttributionCompleted(JNIEnv *env, jclass clazz,
                                                                      jobject result)
{
    // 直接从 JSONObject 中提取 code 与 data，便于快速验证
    if (result != nullptr)
    {
        jclass jsonCls = env->GetObjectClass(result);
        if (jsonCls != nullptr)
        {
            jmethodID midOptInt = env->GetMethodID(jsonCls, "optInt", "(Ljava/lang/String;I)I");
            jmethodID midOptString = env->GetMethodID(jsonCls, "optString",
                                                      "(Ljava/lang/String;Ljava/lang/String;)Ljava/lang/String;");
            if (midOptInt && midOptString)
            {
                jstring jKeyCode = env->NewStringUTF("code");
                jstring jKeyData = env->NewStringUTF("data");
                jstring jEmpty = env->NewStringUTF("");

                jint code = env->CallIntMethod(result, midOptInt, jKeyCode, (jint)-999);
                jstring jDataStr = (jstring)env->CallObjectMethod(result, midOptString, jKeyData,
                                                                  jEmpty);

                std::string dataStr;
                if (jDataStr != nullptr)
                {
                    dataStr = jStringToString(env, jDataStr);
                    env->DeleteLocalRef(jDataStr);
                }
                CCLOG("[SE] nativeOnAttributionCompleted direct parse: code=%d, data=%s",
                      (int)code, dataStr.c_str());

                // 解析 dataStr 为 SEJSONObject 再取值（顶层字段）
                SEJSONObject dataObj = SEJSONObject::toObject(dataStr.c_str());
                auto getString = [&](const char *key) -> std::string
                {
                    auto it = dataObj.propertiesMap.find(key);
                    if (it == dataObj.propertiesMap.end())
                        return std::string();
                    std::string out;
                    SEJSONObject::ValueNode::toStr(it->second, &out);
                    if (!out.empty() && out.front() == '"' && out.back() == '"')
                        out = out.substr(1, out.size() - 2);
                    return out;
                };

                AttributionData attrData;
                attrData.code = (int)code;
                attrData.account_id = getString("account_id");
                attrData.ad_type = getString("ad_type");
                attrData.adcreative_id = getString("adcreative_id");
                attrData.adcreative_name = getString("adcreative_name");
                attrData.adcreative_type = getString("adcreative_type");
                attrData.adgroup_id = getString("adgroup_id");
                attrData.adgroup_name = getString("adgroup_name");
                attrData.adplan_id = getString("adplan_id");
                attrData.adplan_name = getString("adplan_name");
                attrData.attribution_time = getString("attribution_time");
                attrData.attribution_touch_type = getString("attribution_touch_type");
                attrData.attribution_type = getString("attribution_type");
                attrData.callback_id = getString("callback_id");
                attrData.channel_id = getString("channel_id");
                attrData.channel_name = getString("channel_name");
                attrData.click_id = getString("click_id");
                attrData.conversion_id = getString("conversion_id");
                attrData.impression_id = getString("impression_id");
                attrData.install_time = getString("install_time");
                attrData.placement_id = getString("placement_id");
                attrData.report_time = getString("report_time");
                attrData.request_id = getString("request_id");
                attrData.ry_touchpoint_ts = getString("ry_touchpoint_ts");
                attrData.site_id = getString("site_id");
                attrData.site_name = getString("site_name");
                attrData.turl_campaign_id = getString("turl_campaign_id");
                attrData.turl_campaign_name = getString("turl_campaign_name");
                attrData.turl_id = getString("turl_id");

                attrData.origin_data_str = dataStr;

                cocos2d::Director::getInstance()->getScheduler()->performFunctionInCocosThread(
                    [attrData]()
                    {
                        solarengine::SolarEngineAPI::se_onAttributionDataCallback(attrData);
                    });

                env->DeleteLocalRef(jKeyCode);
                env->DeleteLocalRef(jKeyData);
                env->DeleteLocalRef(jEmpty);
            }
            env->DeleteLocalRef(jsonCls);
        }
    }
}

extern "C" JNIEXPORT void JNICALL
Java_com_solar_engine_SolarEngineCocosAPI_nativeOnDeferredDeeplinkCompleted(JNIEnv *env,
                                                                           jclass clazz,
                                                                           jobject result)
{
    if (result != nullptr)
    {
        jclass jsonCls = env->GetObjectClass(result);
        if (jsonCls != nullptr)
        {
            jmethodID midOptInt = env->GetMethodID(jsonCls, "optInt", "(Ljava/lang/String;I)I");
            jmethodID midOptString = env->GetMethodID(jsonCls, "optString",
                                                      "(Ljava/lang/String;Ljava/lang/String;)Ljava/lang/String;");
            if (midOptInt && midOptString)
            {
                jstring jKeyCode = env->NewStringUTF("code");
                jstring jKeyData = env->NewStringUTF("data");
                jstring jEmpty = env->NewStringUTF("");

                jint code = env->CallIntMethod(result, midOptInt, jKeyCode, (jint)-999);
                jstring jDataStr = (jstring)env->CallObjectMethod(result, midOptString, jKeyData,
                                                                  jEmpty);

                std::string dataStr;
                if (jDataStr != nullptr)
                {
                    dataStr = jStringToString(env, jDataStr);
                    env->DeleteLocalRef(jDataStr);
                }
                CCLOG("[SE] nativeOnDeferredDeeplinkCompleted direct parse: code=%d, data=%s",
                      (int)code, dataStr.c_str());

                // 解析 dataStr 为 SEJSONObject 再取值（顶层字段）
                SEJSONObject dataObj = SEJSONObject::toObject(dataStr.c_str());
                auto getString = [&](const char *key) -> std::string
                {
                    auto it = dataObj.propertiesMap.find(key);
                    if (it == dataObj.propertiesMap.end())
                        return std::string();
                    std::string out;
                    SEJSONObject::ValueNode::toStr(it->second, &out);
                    if (!out.empty() && out.front() == '"' && out.back() == '"')
                        out = out.substr(1, out.size() - 2);
                    return out;
                };

                DeferredDeeplinkData deferredDeeplinkData;
                deferredDeeplinkData.code = (int)code;
                deferredDeeplinkData.sedpLink = getString("sedpLink");
                deferredDeeplinkData.turlId = getString("turlId");
                deferredDeeplinkData.sedpUrlscheme = getString("sedpUrlscheme");

                cocos2d::Director::getInstance()->getScheduler()->performFunctionInCocosThread(
                    [deferredDeeplinkData]()
                    {
                        solarengine::SolarEngineAPI::se_onDeferredDeeplinkCallback(
                            deferredDeeplinkData);
                    });

                env->DeleteLocalRef(jKeyCode);
                env->DeleteLocalRef(jKeyData);
                env->DeleteLocalRef(jEmpty);
            }
            env->DeleteLocalRef(jsonCls);
        }
    }
}

extern "C" JNIEXPORT void JNICALL
Java_com_solar_engine_SolarEngineCocosAPI_nativeOnDeeplinkCompleted(JNIEnv *env, jclass clazz,
                                                                   jobject result)
{
    if (result != nullptr)
    {
        jclass jsonCls = env->GetObjectClass(result);
        if (jsonCls != nullptr)
        {
            jmethodID midOptInt = env->GetMethodID(jsonCls, "optInt", "(Ljava/lang/String;I)I");
            jmethodID midOptString = env->GetMethodID(jsonCls, "optString",
                                                      "(Ljava/lang/String;Ljava/lang/String;)Ljava/lang/String;");
            if (midOptInt && midOptString)
            {
                jstring jKeyCode = env->NewStringUTF("code");
                jstring jKeyData = env->NewStringUTF("data");
                jstring jEmpty = env->NewStringUTF("");

                jint code = env->CallIntMethod(result, midOptInt, jKeyCode, (jint)-999);
                jstring jDataStr = (jstring)env->CallObjectMethod(result, midOptString, jKeyData,
                                                                  jEmpty);

                std::string dataStr;
                if (jDataStr != nullptr)
                {
                    dataStr = jStringToString(env, jDataStr);
                    env->DeleteLocalRef(jDataStr);
                }
                CCLOG("[SE] nativeOnDeeplinkCompleted direct parse: code=%d, data=%s", (int)code,
                      dataStr.c_str());

                // 解析 dataStr 为 SEJSONObject 再取值（顶层字段）
                SEJSONObject dataObj = SEJSONObject::toObject(dataStr.c_str());
                auto getString = [&](const char *key) -> std::string
                {
                    auto it = dataObj.propertiesMap.find(key);
                    if (it == dataObj.propertiesMap.end())
                        return std::string();
                    std::string out;
                    SEJSONObject::ValueNode::toStr(it->second, &out);
                    if (!out.empty() && out.front() == '"' && out.back() == '"')
                        out = out.substr(1, out.size() - 2);
                    return out;
                };

                DeeplinkData deeplinkData;
                deeplinkData.code = (int)code;
                deeplinkData.sedpLink = getString("sedpLink");
                deeplinkData.turlId = getString("turlId");
                deeplinkData.from = getString("from");
                deeplinkData.baseUrl = getString("baseUrl");
                deeplinkData.url = getString("url");

                // 解析 customParams 字段
                auto it = dataObj.propertiesMap.find("customParams");
                if (it != dataObj.propertiesMap.end())
                {
                    // 如果 customParams 是对象类型，直接赋值
                    if (it->second.getType() == SEJSONObject::OBJECT)
                    {
                        deeplinkData.customParams = it->second.getObjectData();
                    }
                    else
                    {
                        // 如果不是对象类型，尝试解析为字符串再转换为对象
                        std::string customParamsStr;
                        SEJSONObject::ValueNode::toStr(it->second, &customParamsStr);
                        if (!customParamsStr.empty() && customParamsStr.front() == '"' &&
                            customParamsStr.back() == '"')
                        {
                            customParamsStr = customParamsStr.substr(1, customParamsStr.size() - 2);
                        }
                        if (!customParamsStr.empty())
                        {
                            deeplinkData.customParams = SEJSONObject::toObject(
                                customParamsStr.c_str());
                        }
                    }
                }

                cocos2d::Director::getInstance()->getScheduler()->performFunctionInCocosThread(
                    [deeplinkData]()
                    {
                        solarengine::SolarEngineAPI::se_onDeeplinkCallback(deeplinkData);
                    });

                env->DeleteLocalRef(jKeyCode);
                env->DeleteLocalRef(jKeyData);
                env->DeleteLocalRef(jEmpty);
            }
            env->DeleteLocalRef(jsonCls);
        }
    }
}


#endif