#include "SEJSONObject.h"
#include <sstream>
#include <iostream>

using namespace solarengine;

void SEJSONObject::removeKey(const char *propertyName)
{
    if (!propertyName)
        return;
    std::map<string, ValueNode>::iterator it;
    it = propertiesMap.find(propertyName);
    if (it != propertiesMap.end())
    {
        propertiesMap.erase(it);
    }
}

void SEJSONObject::setJsonObject(const char *propertyName, const SEJSONObject &node)
{
    if (!propertyName)
        return;
    propertiesMap[string(propertyName)] = ValueNode(node);
}

void SEJSONObject::setNumber(const char *propertyName, double value)
{
    if (!propertyName)
        return;
    propertiesMap[string(propertyName)] = ValueNode(value);
}

void SEJSONObject::setNumber(const char *propertyName, int32_t value)
{
    if (!propertyName)
        return;
    propertiesMap[string(propertyName)] = ValueNode(static_cast<int64_t>(value));
}

void SEJSONObject::setNumber(const char *propertyName, int64_t value)
{
    if (!propertyName)
        return;
    propertiesMap[string(propertyName)] = ValueNode(value);
}

void SEJSONObject::setString(const char *propertyName, const char *value)
{
    if (!propertyName || !value)
        return;
    propertiesMap[string(propertyName)] = ValueNode(string(value));
}

void SEJSONObject::setBool(const char *propertyName, bool value)
{
    if (!propertyName)
        return;
    propertiesMap[string(propertyName)] = ValueNode(value);
}

void SEJSONObject::setList(const char *propertyName, const std::vector<string> &value)
{
    if (!propertyName)
        return;
    propertiesMap[string(propertyName)] = ValueNode(value);
}

void SEJSONObject::setList(const char *propertyName, const std::vector<SEJSONObject> &value)
{
    if (!propertyName)
        return;
    propertiesMap[string(propertyName)] = ValueNode(value);
}

void SEJSONObject::setDateTime(const char *propertyName, const time_t seconds, int milliseconds)
{
    if (!propertyName)
        return;
    propertiesMap[string(propertyName)] = ValueNode(seconds, milliseconds);
}

void SEJSONObject::setDateTime(const char *propertyName, const char *value)
{
    if (!propertyName || !value)
        return;
    propertiesMap[string(propertyName)] = ValueNode(string(value));
}

void SEJSONObject::clear()
{
    propertiesMap.clear();
}

void SEJSONObject::dumpNode(const SEJSONObject &node, string *buffer)
{
    *buffer += '{';
    bool first = true;

    for (std::map<string, ValueNode>::const_iterator iterator = node.propertiesMap.begin(); iterator != node.propertiesMap.end(); ++iterator)
    {
        if (first)
        {
            first = false;
        }
        else
        {
            *buffer += ',';
        }
        *buffer += '"' + iterator->first + "\":";
        ValueNode::toStr(iterator->second, buffer);
    }
    *buffer += '}';
}

void SEJSONObject::ValueNode::dumpString(const string &value, string *buffer)
{
    *buffer += '"';
    for (std::string::size_type i = 0; i < value.length(); ++i)
    {
        char c = value[i];
        switch (c)
        {
        case '"':
            *buffer += "\\\"";
            break;
        case '\\':
            *buffer += "\\\\";
            break;
        case '\b':
            *buffer += "\\b";
            break;
        case '\f':
            *buffer += "\\f";
            break;
        case '\n':
            *buffer += "\\n";
            break;
        case '\r':
            *buffer += "\\r";
            break;
        case '\t':
            *buffer += "\\t";
            break;
        default:
            *buffer += c;
            break;
        }
    }
    *buffer += '"';
}

void SEJSONObject::ValueNode::dumpList(const std::vector<string> &value, string *buffer)
{
    *buffer += '[';
    bool first = true;
    for (std::vector<string>::const_iterator iterator = value.begin(); iterator != value.end(); ++iterator)
    {
        if (first)
        {
            first = false;
        }
        else
        {
            *buffer += ',';
        }
        dumpString(*iterator, buffer);
    }
    *buffer += ']';
}

void SEJSONObject::ValueNode::dumpList(const std::vector<SEJSONObject> &value, string *buffer)
{
    *buffer += '[';
    bool first = true;
    for (std::vector<SEJSONObject>::const_iterator iterator = value.begin(); iterator != value.end(); ++iterator)
    {
        if (first)
        {
            first = false;
        }
        else
        {
            *buffer += ',';
        }
        dumpNode(*iterator, buffer);
    }
    *buffer += ']';
}

#if defined(__linux__)
#define TD_SDK_LOCALTIME(seconds, now) localtime_r((seconds), (now))
#elif defined(__APPLE__)
#define TD_SDK_LOCALTIME(seconds, now) localtime_r((seconds), (now))
#elif defined(_WIN32)
#define TD_SDK_LOCALTIME(seconds, now) localtime_s((now), (seconds))
#define snprintf sprintf_s
#endif

void SEJSONObject::ValueNode::dumpDateTime(const time_t &seconds, int milliseconds, string *buffer)
{
    struct tm tm = {};
    TD_SDK_LOCALTIME(&seconds, &tm);
    char buff[64];
    snprintf(buff, sizeof(buff), "\"%04d-%02d-%02d %02d:%02d:%02d.%03d\"",
             tm.tm_year + 1900,
             tm.tm_mon + 1,
             tm.tm_mday,
             tm.tm_hour,
             tm.tm_min,
             tm.tm_sec,
             milliseconds);
    *buffer += buff;
}

string SEJSONObject::toJson(const SEJSONObject &node)
{
    string buffer;
    dumpNode(node, &buffer);
    return buffer;
}

static void skipSpaces(const char *&p)
{
    while (*p == ' ' || *p == '\t' || *p == '\n' || *p == '\r')
    {
        ++p;
    }
}

static string parseJSONString(const char *&p)
{
    string out;
    if (*p != '"')
        return out;
    ++p; // skip opening quote
    while (*p && *p != '"')
    {
        if (*p == '\\' && *(p + 1))
        {
            // simple escape handling
            ++p;
            char c = *p;
            switch (c)
            {
            case '"':
                out += '"';
                break;
            case '\\':
                out += '\\';
                break;
            case '/':
                out += '/';
                break;
            case 'b':
                out += '\b';
                break;
            case 'f':
                out += '\f';
                break;
            case 'n':
                out += '\n';
                break;
            case 'r':
                out += '\r';
                break;
            case 't':
                out += '\t';
                break;
            default:
                out += c;
                break;
            }
            ++p;
            continue;
        }
        out += *p;
        ++p;
    }
    if (*p == '"')
        ++p; // skip closing quote
    return out;
}

static string captureBalanced(const char *&p, char openCh, char closeCh)
{
    string out;
    int depth = 0;
    if (*p != openCh)
        return out;
    do
    {
        char c = *p;
        out += c;
        if (c == openCh)
            depth++;
        else if (c == closeCh)
            depth--;
        ++p;
    } while (*p && depth > 0);
    return out;
}

SEJSONObject SEJSONObject::toObject(const char *stringJson)
{
    SEJSONObject obj;
    if (stringJson == NULL)
        return obj;
    const char *p = stringJson;
    skipSpaces(p);
    if (*p != '{')
        return obj;
    ++p; // skip '{'
    skipSpaces(p);
    while (*p && *p != '}')
    {
        skipSpaces(p);
        // key
        if (*p != '"')
            break;
        string key = parseJSONString(p);
        skipSpaces(p);
        if (*p != ':')
            break;
        ++p; // skip ':'
        skipSpaces(p);
        // value
        if (*p == '"')
        {
            string val = parseJSONString(p);
            obj.setString(key.c_str(), val.c_str());
        }
        else if ((*p == '-') || (*p >= '0' && *p <= '9'))
        {
            const char *start = p;
            while ((*p == '-') || (*p == '+') || (*p == '.') || (*p >= '0' && *p <= '9') || (*p == 'e') || (*p == 'E'))
                ++p;
            std::string numStr(start, p - start);
            // prefer integer when possible
            if (numStr.find_first_of(".eE") != std::string::npos)
            {
                obj.setNumber(key.c_str(), atof(numStr.c_str()));
            }
            else
            {
                obj.setNumber(key.c_str(), (int64_t)atoll(numStr.c_str()));
            }
        }
        else if (*p == '{')
        {
            const char *start = p;
            string raw = captureBalanced(p, '{', '}');
            obj.setString(key.c_str(), raw.c_str());
        }
        else if (*p == '[')
        {
            const char *start = p;
            string raw = captureBalanced(p, '[', ']');
            obj.setString(key.c_str(), raw.c_str());
        }
        else if (strncmp(p, "true", 4) == 0)
        {
            obj.setBool(key.c_str(), true);
            p += 4;
        }
        else if (strncmp(p, "false", 5) == 0)
        {
            obj.setBool(key.c_str(), false);
            p += 5;
        }
        else if (strncmp(p, "null", 4) == 0)
        {
            obj.setString(key.c_str(), "");
            p += 4;
        }
        // skip spaces and comma
        skipSpaces(p);
        if (*p == ',')
        {
            ++p;
            skipSpaces(p);
        }
    }
    // consume closing '}' if present
    if (*p == '}')
        ++p;
    return obj;
}

SEJSONObject::ValueNode::ValueNode(double value) : nodeType(NUMBER)
{
    valueData.numberValue = value;
}

SEJSONObject::ValueNode::ValueNode(int64_t value) : nodeType(INT)
{
    valueData.intValue = value;
}

SEJSONObject::ValueNode::ValueNode(const string &value) : nodeType(STRING), stringData(value) {}

SEJSONObject::ValueNode::ValueNode(bool value) : nodeType(BOOL1)
{
    valueData.boolValue = value;
}

SEJSONObject::ValueNode::ValueNode(const std::vector<string> &value) : nodeType(LIST), listData(value) {}

SEJSONObject::ValueNode::ValueNode(time_t seconds, int milliseconds) : nodeType(DATETIME)
{
    valueData.datetimeValue.seconds = seconds;
    valueData.datetimeValue.milliseconds = milliseconds;
}

SEJSONObject::ValueNode::ValueNode(const SEJSONObject &value) : nodeType(OBJECT), objectData(value) {}

SEJSONObject::ValueNode::ValueNode(const std::vector<SEJSONObject> &value) : nodeType(OBJECTS), objectDatas(value) {}

void SEJSONObject::ValueNode::toStr(const SEJSONObject::ValueNode &node, string *buffer)
{
    switch (node.nodeType)
    {
    case NUMBER:
        dumpNumber(node.valueData.numberValue, buffer);
        break;
    case INT:
        dumpNumber(node.valueData.intValue, buffer);
        break;
    case STRING:
        dumpString(node.stringData, buffer);
        break;
    case LIST:
        dumpList(node.listData, buffer);
        break;
    case BOOL1:
        *buffer += (node.valueData.boolValue ? "true" : "false");
        break;
    case DATETIME:
        dumpDateTime(node.valueData.datetimeValue.seconds,
                     node.valueData.datetimeValue.milliseconds, buffer);
        break;
    case OBJECT:
        dumpNode(node.objectData, buffer);
        break;
    case OBJECTS:
        dumpList(node.objectDatas, buffer);
        break;
    default:
        break;
    }
}

void SEJSONObject::ValueNode::dumpNumber(double value, string *buffer)
{
    std::ostringstream buf;
    buf << value;
    *buffer += buf.str();
}

void SEJSONObject::ValueNode::dumpNumber(int64_t value, string *buffer)
{
    std::ostringstream buf;
    buf << value;
    *buffer += buf.str();
}

void SEJSONObject::mergeFrom(const SEJSONObject &anotherNode)
{
    for (std::map<string, ValueNode>::const_iterator
             iterator = anotherNode.propertiesMap.begin();
         iterator != anotherNode.propertiesMap.end(); ++iterator)
    {
        propertiesMap[iterator->first] = iterator->second;
    }
}
