
#if CC_TARGET_PLATFORM == CC_PLATFORM_IOS

#import <SolarEngineSDK/SolarEngineSDK.h>
#import <SESDKRemoteConfig/SESDKRemoteConfig.h>

#if  __has_include(<SolarEngineSDK/SESDKForCN.h>)
    #import <SolarEngineSDK/SESDKForCN.h>
#endif
#if  __has_include(<SolarEngineSDK/SESDKForUS.h>)
    #import <SolarEngineSDK/SESDKForUS.h>
#endif

#import <Foundation/Foundation.h>
#include "../Common/SolarEngineAPI.h"
using  namespace solarengine;
using  namespace std;

NSString * const SEAppRegisterPropertyType                     = @"registerType";
NSString * const SEAppRegisterPropertyStatus                   = @"registerStatus";
NSString * const SEAppKeyCustomProperties                      = @"customProperties";
NSString * const SEAppKeyCustomEventName                       = @"_customEventName";

static NSString *NSStringFromCString(const char* cstr){
    return cstr ? [NSString stringWithUTF8String:cstr] : nil;
}

static char *CStringFromNSString(NSString *string) {
    return (char *)[string UTF8String];
}

static NSDictionary *NSDictionaryFromJSONObject(const SEJSONObject &jsonObject) {
    string json = SEJSONObject::toJson(jsonObject);
    NSString *string = NSStringFromCString(json.c_str());
    NSData *data = [string dataUsingEncoding:NSUTF8StringEncoding];
    
    if (!data) return nil;
    return [NSJSONSerialization JSONObjectWithData:data options:kNilOptions error:nil];
}

static std::vector<string> ListFromNSArray(NSArray* array)
{
    vector<string> list;
    for(NSObject *value in array)
    {
        NSString *stringValue = nil;
        if([value isKindOfClass:[NSString class]])
        {
            stringValue = (NSString *)value;
        }
        else if([value isKindOfClass:[NSNumber class]])
        {
            NSNumber *number = (NSNumber *)value;
            if (strcmp([number objCType], @encode(BOOL)) == 0)
            {
                stringValue = number.boolValue ? @"true" : @"false";
            }
            else
            {
                stringValue = [number stringValue];
            }
        }
        else if([value isKindOfClass:[NSArray class]] || [value isKindOfClass:[NSDictionary class]])
        {
            // 将嵌套数组序列化为 JSON 字符串
            NSError *error = nil;
            NSData *jsonData = [NSJSONSerialization dataWithJSONObject:(NSArray *)value options:0 error:&error];
            if (jsonData && !error)
            {
                stringValue = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
            }
            else
            {
                stringValue = [value description];
            }
        }
        else
        {
            stringValue = [value description];
        }

        if (stringValue)
        {
            list.push_back(CStringFromNSString(stringValue));
        }
        else
        {
            list.push_back("");
        }
    }
    return list;
}

static SEJSONObject JSONObjectFromNSDictionary(NSDictionary* dic)
{
    SEJSONObject object;
    for(int i = 0 ; i <dic.allKeys.count ; i++)
    {
        NSString* key = dic.allKeys[i];
        NSObject* value = dic[key];
        if([value isKindOfClass:[NSString class]])
        {
            object.setString(CStringFromNSString(key), CStringFromNSString((NSString*)value));
        }else if([value isKindOfClass:[NSNumber class]])
        {
            NSNumber *number = (NSNumber*)value;
            if (strcmp([number objCType], @encode(long)) == 0)
            {
                int64_t num = [number longValue];
                object.setNumber(CStringFromNSString(key),num);
            }else if (strcmp([number objCType], @encode(long long)) == 0)
            {
                int64_t num = [number longLongValue];
                object.setNumber(CStringFromNSString(key), num);
            }
            else if (strcmp([number objCType], @encode(int)) == 0)
            {
                int32_t num = [number intValue];
                object.setNumber(CStringFromNSString(key), num);
            }
            else if (strcmp([number objCType], @encode(BOOL)) == 0)
            {
                object.setNumber(CStringFromNSString(key), number.boolValue);
            }else if (strcmp([number objCType], @encode(double)) == 0)
            {
                object.setNumber(CStringFromNSString(key), number.doubleValue);
            }
            else if (strcmp([number objCType], @encode(float)) == 0)
            {
                object.setNumber(CStringFromNSString(key), number.floatValue);
            }
        }else if([value isKindOfClass:[NSArray class]])
        {
            // 将数组序列化为 JSON 字符串
            NSError *error = nil;
            NSData *jsonData = [NSJSONSerialization dataWithJSONObject:(NSArray *)value options:0 error:&error];
            if (jsonData && !error)
            {
                NSString *jsonStr = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
                object.setString(CStringFromNSString(key), CStringFromNSString(jsonStr));
            }
            else
            {
                object.setString(CStringFromNSString(key), CStringFromNSString([value description]));
            }
        }else if([value isKindOfClass:[NSDictionary class]])
        {
            // 将字典序列化为 JSON 字符串
            NSError *error = nil;
            NSData *jsonData = [NSJSONSerialization dataWithJSONObject:(NSDictionary *)value options:0 error:&error];
            if (jsonData && !error)
            {
                NSString *jsonStr = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
                object.setString(CStringFromNSString(key), CStringFromNSString(jsonStr));
            }
            else
            {
                object.setString(CStringFromNSString(key), CStringFromNSString([value description]));
            }
        }
    }
    return object;
}

static char *CStringFromNSDictionary(NSDictionary *dic) {
    if (!dic) return nil;
    NSData *jsonData = [NSJSONSerialization dataWithJSONObject:dic options:NSJSONWritingPrettyPrinted error:nil];
    
    if (!jsonData) return nil;
    NSString *jsonStr = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
    return (char *)[jsonStr UTF8String];
}


@interface SEWrapperManager : NSObject

@property (nonatomic,copy)NSString *sub_lib_version;
@property (nonatomic,copy)NSString *sdk_type;

+ (SEWrapperManager *)sharedInstance;

@end
@implementation SEWrapperManager

+ (SEWrapperManager *)sharedInstance {
    static SEWrapperManager * instance = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        instance = [[self alloc] init];
    });
    return instance;
}

- (instancetype)init
{
    self = [super init];
    if (self) {
        _sdk_type = @"cocos2d-x";
    }
    return self;
}

@end

namespace solarengine {
    std::function<void(int)> SolarEngineAPI::se_initCallback = nullptr;
    std::function<void(const AttributionData &)> SolarEngineAPI::se_attributionDataCallback = nullptr;
    std::function<void(const DeferredDeeplinkData &)> SolarEngineAPI::se_deferredDeeplinkCallback = nullptr;
    std::function<void(const DeeplinkData &)> SolarEngineAPI::se_deeplinkCallback = nullptr;

    void SolarEngineAPI::setInitCallback(std::function<void(int)> callback)
    {
        se_initCallback = callback;
    }
    void SolarEngineAPI::setAttributionDataCallback(std::function<void(const AttributionData &)> callback)
    {
        se_attributionDataCallback = callback;
    }
    void SolarEngineAPI::setDeferredDeeplinkCallback(std::function<void(const DeferredDeeplinkData &)> callback)
    {
        se_deferredDeeplinkCallback = callback;
    }
    void SolarEngineAPI::setDeeplinkCallback(std::function<void(const DeeplinkData &)> callback)
    {
        se_deeplinkCallback = callback;
    }

    // 基础初始化方法
    void SolarEngineAPI::preInit(string appKey)
    {
        NSString* _appid = NSStringFromCString(appKey.c_str());
        [[SolarEngineSDK sharedInstance] preInitWithAppKey:_appid];
    }

    void SolarEngineAPI::init(string appKey, SECConfig config)
    {
        NSString* _appid = NSStringFromCString(appKey.c_str());
        SEConfig *_config = [[SEConfig alloc] init];
        _config.logEnabled = config.enableLog;
        _config.isDebugModel = config.isDebugModel;
        _config.isGDPRArea = config.isGDPRArea;
        _config.setCoppaEnabled = config.setCoppaEnabled;
        _config.setKidsAppEnabled = config.setKidsAppEnabled;
        _config.enable2GReporting = config.enable2GReporting;
        _config.attAuthorizationWaitingInterval = config.attAuthorizationWaitingInterval;
        _config.enableDeferredDeeplink = config.enableDeferredDeeplink;
                
#if  __has_include(<SolarEngineSDK/SESDKForCN.h>)
        _config.caid = NSStringFromCString(config.caid.c_str());
#endif
    
#if  __has_include(<SolarEngineSDK/SESDKForUS.h>)
        _config.enableODMInfo = config.enableODMInfo;
#endif
        
        [SEWrapperManager sharedInstance].sub_lib_version = NSStringFromCString(SE_COCOS_LIB_VERSION.c_str());
        
        // 设置远程配置
        if (config.remoteConfig.enable) {
            SERemoteConfig *remoteConfig = [[SERemoteConfig alloc] init];
            remoteConfig.enable = config.remoteConfig.enable;
            remoteConfig.logEnabled = config.enableLog;
            remoteConfig.mergeType = (SERCMergeType)config.remoteConfig.mergeType;
            remoteConfig.customIDEventProperties = NSDictionaryFromJSONObject(config.remoteConfig.customIDEventProperties);
            remoteConfig.customIDUserProperties = NSDictionaryFromJSONObject(config.remoteConfig.customIDUserProperties);
            _config.remoteConfig = remoteConfig;
        }
        
        // 设置私有化部署配置
        if (config.customDomain.enable) {
            SECustomDomain *customDomain = [[SECustomDomain alloc] init];
            customDomain.enable = config.customDomain.enable;
            customDomain.receiverDomain = NSStringFromCString(config.customDomain.receiverDomain.c_str());
            customDomain.ruleDomain = NSStringFromCString(config.customDomain.ruleDomain.c_str());
            customDomain.receiverTcpHost = NSStringFromCString(config.customDomain.receiverTcpHost.c_str());
            customDomain.ruleTcpHost = NSStringFromCString(config.customDomain.ruleTcpHost.c_str());
            customDomain.gatewayTcpHost = NSStringFromCString(config.customDomain.gatewayTcpHost.c_str());
            _config.customDomain = customDomain;
        }
        
        [[SolarEngineSDK sharedInstance] setAttributionCallback:^(int code, NSDictionary * _Nullable attributionData) {
            
            AttributionData attribution;
            attribution.code = code;
            if (code == 0) {
                attribution.account_id = CStringFromNSString(attributionData[@"account_id"]);
                attribution.ad_type = CStringFromNSString(attributionData[@"ad_type"]);
                attribution.adcreative_id = CStringFromNSString(attributionData[@"adcreative_id"]);
                attribution.adcreative_name = CStringFromNSString(attributionData[@"adcreative_name"]);
                attribution.adcreative_type = CStringFromNSString(attributionData[@"adcreative_type"]);
                attribution.adgroup_id = CStringFromNSString(attributionData[@"adgroup_id"]);
                attribution.adgroup_name = CStringFromNSString(attributionData[@"adgroup_name"]);
                attribution.adplan_id = CStringFromNSString(attributionData[@"adplan_id"]);
                attribution.adplan_name = CStringFromNSString(attributionData[@"adplan_name"]);
                attribution.attribution_time = CStringFromNSString(attributionData[@"attribution_time"]);
                attribution.attribution_touch_type = CStringFromNSString(attributionData[@"attribution_touch_type"]);
                attribution.attribution_type = CStringFromNSString(attributionData[@"attribution_type"]);
                attribution.callback_id = CStringFromNSString(attributionData[@"callback_id"]);
                attribution.channel_id = CStringFromNSString(attributionData[@"channel_id"]);
                attribution.channel_name = CStringFromNSString(attributionData[@"channel_name"]);
                attribution.click_id = CStringFromNSString(attributionData[@"click_id"]);
                attribution.conversion_id = CStringFromNSString(attributionData[@"conversion_id"]);
                attribution.impression_id = CStringFromNSString(attributionData[@"impression_id"]);
                attribution.install_time = CStringFromNSString(attributionData[@"install_time"]);
                attribution.placement_id = CStringFromNSString(attributionData[@"placement_id"]);
                attribution.report_time = CStringFromNSString(attributionData[@"report_time"]);
                attribution.request_id = CStringFromNSString(attributionData[@"request_id"]);
                attribution.ry_touchpoint_ts = CStringFromNSString(attributionData[@"ry_touchpoint_ts"]);
                attribution.site_id = CStringFromNSString(attributionData[@"site_id"]);
                attribution.site_name = CStringFromNSString(attributionData[@"site_name"]);
                attribution.turl_campaign_id = CStringFromNSString(attributionData[@"turl_campaign_id"]);
                attribution.turl_campaign_name = CStringFromNSString(attributionData[@"turl_campaign_name"]);
                attribution.turl_campaign_name = CStringFromNSString(attributionData[@"turl_campaign_name"]);
                
                NSData *data = [NSJSONSerialization dataWithJSONObject:attributionData options:0 error:nil];
                attribution.origin_data_str = CStringFromNSString([[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding]);
                
            }
            if (se_attributionDataCallback) {
                se_attributionDataCallback(attribution);
            }
            
        }];
        
        [[SolarEngineSDK sharedInstance] setDeepLinkCallback:^(int code, SEDeeplinkInfo * _Nullable deeplinkInfo) {
            
            DeeplinkData deeplink;
            deeplink.code = code;
            if (code == 0) {
                deeplink.turlId = CStringFromNSString(deeplinkInfo.turlId);
                deeplink.sedpLink = CStringFromNSString(deeplinkInfo.sedpLink);
                deeplink.from = CStringFromNSString(deeplinkInfo.from);
                deeplink.baseUrl = CStringFromNSString(deeplinkInfo.baseUrl);
                deeplink.url = CStringFromNSString(deeplinkInfo.url);
                deeplink.customParams = JSONObjectFromNSDictionary(deeplinkInfo.customParams);
            }
            
            if (se_deeplinkCallback) {
                se_deeplinkCallback(deeplink);
            }
            
        }];
        [[SolarEngineSDK sharedInstance] setDeferredDeepLinkCallbackWithSuccess:^(SEDeferredDeeplinkInfo * _Nullable deeplinkInfo) {
            
            DeferredDeeplinkData deeplink;
            deeplink.code = 0;
            deeplink.turlId = CStringFromNSString(deeplinkInfo.turlId);
            deeplink.sedpLink = CStringFromNSString(deeplinkInfo.sedpLink);
            deeplink.sedpUrlscheme = CStringFromNSString(deeplinkInfo.sedpUrlscheme);
            
            if (se_deferredDeeplinkCallback) {
                se_deferredDeeplinkCallback(deeplink);
            }
        } fail:^(NSError * _Nullable error) {
            
            DeferredDeeplinkData deeplink;
            deeplink.code = (int)error.code;
            if (se_deferredDeeplinkCallback) {
                se_deferredDeeplinkCallback(deeplink);
            }
        }];

        [[SolarEngineSDK sharedInstance] setInitCompletedCallback:^(int code) {
            if (se_initCallback)
            {
                se_initCallback(code);
            }
        }];
        
        [[SolarEngineSDK sharedInstance] startWithAppKey:_appid config:_config];
        
    }

    void SolarEngineAPI::setGDPRArea(bool isGDPRArea)
    {
        [[SolarEngineSDK sharedInstance] setGDPRArea:isGDPRArea];
    }

    // 事件追踪方法
    void SolarEngineAPI::track(SECustomEvent *event)
    {
        if (event == NULL) {
            printf("track SECustomEvent, event is null ");
            return;
        }
        
        NSString* eventName = NSStringFromCString((*event).eventName.c_str());
        NSDictionary* customProperties = NSDictionaryFromJSONObject((*event).customProperties);
        NSDictionary* presetProperties = NSDictionaryFromJSONObject((*event).presetProperties);
        
        [[SolarEngineSDK sharedInstance] track:eventName withCustomProperties:customProperties withPresetProperties:presetProperties];
    }

    void SolarEngineAPI::track(SEPurchaseEvent *event)
    {
        if (event == NULL) {
            printf("track SEIAPEvent, event is null ");
            return;
        }
        
        SEIAPEventAttribute *eventAttribute = [[SEIAPEventAttribute alloc] init];
        
        eventAttribute.productID = NSStringFromCString((*event).productId.c_str());
        eventAttribute.productName = NSStringFromCString((*event).productName.c_str());
        eventAttribute.productCount = (*event).productNum;
        eventAttribute.orderId = NSStringFromCString((*event).orderId.c_str());
        eventAttribute.payAmount = (*event).payAmount;
        eventAttribute.currencyType = NSStringFromCString((*event).currencyType.c_str());
        eventAttribute.payType = NSStringFromCString((*event).payType.c_str());
        eventAttribute.payStatus = (SolarEngineIAPStatus)(*event).payStatus;
        eventAttribute.failReason = NSStringFromCString((*event).failReason.c_str());
        eventAttribute.customProperties = NSDictionaryFromJSONObject((*event).customProperties);
        
        [[SolarEngineSDK sharedInstance] trackIAPWithAttributes:eventAttribute];
    }

    void SolarEngineAPI::track(SEAdImpressionEvent *event)
    {
        if (event == NULL) {
            printf("track SEAdImpressionEvent, event is null ");
            return;
        }
        
        SEAdImpressionEventAttribute *eventAttribute = [[SEAdImpressionEventAttribute alloc] init];
        
        eventAttribute.adType = (*event).adType;
        eventAttribute.adNetworkPlatform = NSStringFromCString((*event).adNetworkPlatform.c_str());
        eventAttribute.adNetworkAppID = NSStringFromCString((*event).adNetworkAppID.c_str());
        eventAttribute.adNetworkPlacementID = NSStringFromCString((*event).adNetworkPlacementID.c_str());
        eventAttribute.currency = NSStringFromCString((*event).currency.c_str());
        eventAttribute.ecpm = (*event).ecpm;
        eventAttribute.mediationPlatform = NSStringFromCString((*event).mediationPlatform.c_str());
        eventAttribute.rendered = (*event).rendered;
        eventAttribute.customProperties = NSDictionaryFromJSONObject((*event).customProperties);
        
        [[SolarEngineSDK sharedInstance] trackAdImpressionWithAttributes:eventAttribute];
    }

    void SolarEngineAPI::track(SEAdClickEvent *event)
    {
        if (event == NULL) {
            printf("track SEAdClickEvent, event is null ");
            return;
        }
        
        SEAdClickEventAttribute *eventAttribute = [[SEAdClickEventAttribute alloc] init];
        
        eventAttribute.adType = (*event).adType;
        eventAttribute.adNetworkPlatform = NSStringFromCString((*event).adNetworkPlatform.c_str());
        eventAttribute.adNetworkPlacementID = NSStringFromCString((*event).adNetworkPlacementID.c_str());
        eventAttribute.mediationPlatform = NSStringFromCString((*event).mediationPlatform.c_str());
        eventAttribute.customProperties = NSDictionaryFromJSONObject((*event).customProperties);
        
        [[SolarEngineSDK sharedInstance] trackAdClickWithAttributes:eventAttribute];
    }

    void SolarEngineAPI::track(SEAppAttrEvent *event)
    {
        if (event == NULL) {
            printf("track SEAppAttrEvent, event is null ");
            return;
        }
        
        SEAppAttrEventAttribute *eventAttribute = [[SEAppAttrEventAttribute alloc] init];
        
        eventAttribute.adNetwork = NSStringFromCString((*event).adNetwork.c_str());
        eventAttribute.subChannel = NSStringFromCString((*event).subChannel.c_str());
        eventAttribute.adAccountID = NSStringFromCString((*event).adAccountId.c_str());
        eventAttribute.adAccountName = NSStringFromCString((*event).adAccountName.c_str());
        eventAttribute.adCampaignID = NSStringFromCString((*event).adCampaignId.c_str());
        eventAttribute.adCampaignName = NSStringFromCString((*event).adCampaignName.c_str());
        eventAttribute.adOfferID = NSStringFromCString((*event).adOfferId.c_str());
        eventAttribute.adOfferName = NSStringFromCString((*event).adOfferName.c_str());
        eventAttribute.adCreativeID = NSStringFromCString((*event).adCreativeId.c_str());
        eventAttribute.adCreativeName = NSStringFromCString((*event).adCreativeName.c_str());
        eventAttribute.attributionPlatform = NSStringFromCString((*event).attributionPlatform.c_str());
        eventAttribute.customProperties = NSDictionaryFromJSONObject((*event).customProperties);
        
        [[SolarEngineSDK sharedInstance] trackAppAttrWithAttributes:eventAttribute];
    }

    void SolarEngineAPI::track(SERegisterEvent *event)
    {
        if (event == NULL) {
            printf("track SERegisterEvent, event is null ");
            return;
        }

        SERegisterEventAttribute *eventAttribute = [[SERegisterEventAttribute alloc] init];
        
        eventAttribute.registerType = NSStringFromCString((*event).registerType.c_str());
        eventAttribute.registerStatus = NSStringFromCString((*event).registerStatus.c_str());
        eventAttribute.customProperties = NSDictionaryFromJSONObject((*event).customProperties);
        
        [[SolarEngineSDK sharedInstance] trackRegisterWithAttributes:eventAttribute];
    }

    void SolarEngineAPI::track(SELoginEvent *event)
    {
        if (event == NULL) {
            printf("track SELoginEvent, event is null ");
            return;
        }
        
        SELoginEventAttribute *eventAttribute = [[SELoginEventAttribute alloc] init];
        
        eventAttribute.loginType = NSStringFromCString((*event).loginType.c_str());
        eventAttribute.loginStatus = NSStringFromCString((*event).loginStatus.c_str());
        eventAttribute.customProperties = NSDictionaryFromJSONObject((*event).customProperties);
        
        [[SolarEngineSDK sharedInstance] trackLoginWithAttributes:eventAttribute];
    }

    void SolarEngineAPI::track(SEOrderEvent *event)
    {
        if (event == NULL) {
            printf("track SEOrderEvent, event is null ");
            return;
        }
        
        SEOrderEventAttribute *eventAttribute = [[SEOrderEventAttribute alloc] init];
        
        eventAttribute.orderID = NSStringFromCString((*event).orderId.c_str());
        eventAttribute.payAmount = (*event).payAmount;
        eventAttribute.currencyType = NSStringFromCString((*event).currencyType.c_str());
        eventAttribute.payType = NSStringFromCString((*event).payType.c_str());
        eventAttribute.status = NSStringFromCString((*event).status.c_str());
        eventAttribute.customProperties = NSDictionaryFromJSONObject((*event).customProperties);
        
        [[SolarEngineSDK sharedInstance] trackOrderWithAttributes:eventAttribute];
    }

    void SolarEngineAPI::trackFirstEvent(string eventName, string firstCheckId, const SEJSONObject &properties)
    {
        
        NSString *eventNameStr = NSStringFromCString(eventName.c_str());
        
        NSDictionary *options = NSDictionaryFromJSONObject(properties);
        
        if ([eventNameStr isEqualToString:@"_appReg"]) {
            
            SERegisterEventAttribute *eventAttribute = [[SERegisterEventAttribute alloc] init];
            
            eventAttribute.registerType = [options objectForKey:SEAppRegisterPropertyType];
            eventAttribute.registerStatus = [options objectForKey:SEAppRegisterPropertyStatus];
            eventAttribute.customProperties = [options objectForKey:SEAppKeyCustomProperties];
            
            eventAttribute.firstCheckId = NSStringFromCString(firstCheckId.c_str());
            
            [[SolarEngineSDK sharedInstance] trackFirstEvent:eventAttribute];

        } else {
            
            if ([eventNameStr hasPrefix:@"_"]) {
                NSLog(@"[SolarEngine Cocos] Error: trackFirstEvent api only support _appReg event and custom event.");
            }
            
            SECustomEventAttribute *eventAttribute = [[SECustomEventAttribute alloc] init];
            eventAttribute.firstCheckId = NSStringFromCString(firstCheckId.c_str());
            eventAttribute.eventName = eventNameStr;
            eventAttribute.customProperties = [options objectForKey:SEAppKeyCustomProperties];
            [[SolarEngineSDK sharedInstance] trackFirstEvent:eventAttribute];
        }
    }

    // 事件时长追踪
    void SolarEngineAPI::eventStart(string eventName)
    {
        NSString* _eventName = NSStringFromCString(eventName.c_str());
        [[SolarEngineSDK sharedInstance] eventStart:_eventName];
    }

    void SolarEngineAPI::eventFinish(string eventName, const SEJSONObject& properties)
    {
        NSString* _eventName = NSStringFromCString(eventName.c_str());
        NSDictionary* _properties = NSDictionaryFromJSONObject(properties);
        [[SolarEngineSDK sharedInstance] eventFinish:_eventName properties:_properties];
    }

    // 预置事件属性
    void SolarEngineAPI::setPresetEvent(SECPresetEventType eventType, const SEJSONObject& properties)
    {
        NSDictionary* _properties = NSDictionaryFromJSONObject(properties);
        
        SEPresetEventType type = SEPresetEventTypeAppAll;
        if (eventType == AppInstall) {
            type = SEPresetEventTypeAppInstall;
        } else if (eventType == AppStart) {
            type = SEPresetEventTypeAppStart;
        } else if (eventType == AppEnd) {
            type = SEPresetEventTypeAppEnd;
        }
        
        [[SolarEngineSDK sharedInstance] setPresetEvent:type withProperties:_properties];
    }

    SEPresetProperties SolarEngineAPI::getPresetProperties()
    {
        NSDictionary* properties = [[SolarEngineSDK sharedInstance] getPresetProperties];
        SEPresetProperties preset;
        if (!properties) {
            return preset;
        }

        auto getString = ^std::string(NSString *key) {
            id v = properties[key];
            if (!v) return std::string();
            NSString *s = [v isKindOfClass:NSString.class] ? (NSString *)v : [v description];
            return CStringFromNSString(s);
        };
        auto getInt = ^int(NSString *key) {
            id v = properties[key];
            if (!v) return 0;
            if ([v isKindOfClass:NSNumber.class]) return [(NSNumber *)v intValue];
            return [[v description] intValue];
        };
        auto getDouble = ^double (NSString *key) {
            id v = properties[key];
            if (!v) return 0.0;
            if ([v isKindOfClass:NSNumber.class]) return [(NSNumber *)v doubleValue];
            return [[v description] doubleValue];
        };

        preset.appkey = getString(@"_appkey");
        preset.distinctId = getString(@"_distinct_id");
        preset.accountId = getString(@"_account_id");
        preset.visitorId = getString(@"_visitor_id");
        preset.sessionId = getString(@"_session_id");
        preset.idfa = getString(@"_idfa");
        preset.idfv = getString(@"_idfv");
        preset.ua = getString(@"_ua");
        preset.language = getString(@"_language");
        preset.timeZone = getString(@"_time_zone");
        preset.manufacturer = getString(@"_manufacturer");
        preset.platform = getString(@"_platform");
        preset.osVersion = getString(@"_os_version");
        preset.screenHeight = getInt(@"_screen_height");
        preset.screenWidth = getInt(@"_screen_width");
        preset.deviceModel = getString(@"_device_model");
        preset.deviceType = getInt(@"_device_type");
        preset.appVersion = getString(@"_app_version");
        preset.appVersionCode = getInt(@"_app_version_code");
        preset.packageName = getString(@"_package_name");
        preset.appName = getString(@"_app_name");
        preset.channel = getString(@"_channel");
        preset.lib = getString(@"_lib");
        preset.libVersion = getString(@"_lib_version");

        return preset;
    }

    AttributionData SolarEngineAPI::getAttributionData()
    {
        NSDictionary* attributionData = [[SolarEngineSDK sharedInstance] getAttributionData];
        AttributionData attribution;
        
        if (!attributionData || attributionData.allKeys.count == 0) {
            attribution.code = -1;
            return attribution;
        } else {
            attribution.code = 0;
            attribution.account_id = CStringFromNSString(attributionData[@"account_id"]);
            attribution.ad_type = CStringFromNSString(attributionData[@"ad_type"]);
            attribution.adcreative_id = CStringFromNSString(attributionData[@"adcreative_id"]);
            attribution.adcreative_name = CStringFromNSString(attributionData[@"adcreative_name"]);
            attribution.adcreative_type = CStringFromNSString(attributionData[@"adcreative_type"]);
            attribution.adgroup_id = CStringFromNSString(attributionData[@"adgroup_id"]);
            attribution.adgroup_name = CStringFromNSString(attributionData[@"adgroup_name"]);
            attribution.adplan_id = CStringFromNSString(attributionData[@"adplan_id"]);
            attribution.adplan_name = CStringFromNSString(attributionData[@"adplan_name"]);
            attribution.attribution_time = CStringFromNSString(attributionData[@"attribution_time"]);
            attribution.attribution_touch_type = CStringFromNSString(attributionData[@"attribution_touch_type"]);
            attribution.attribution_type = CStringFromNSString(attributionData[@"attribution_type"]);
            attribution.callback_id = CStringFromNSString(attributionData[@"callback_id"]);
            attribution.channel_id = CStringFromNSString(attributionData[@"channel_id"]);
            attribution.channel_name = CStringFromNSString(attributionData[@"channel_name"]);
            attribution.click_id = CStringFromNSString(attributionData[@"click_id"]);
            attribution.conversion_id = CStringFromNSString(attributionData[@"conversion_id"]);
            attribution.impression_id = CStringFromNSString(attributionData[@"impression_id"]);
            attribution.install_time = CStringFromNSString(attributionData[@"install_time"]);
            attribution.placement_id = CStringFromNSString(attributionData[@"placement_id"]);
            attribution.report_time = CStringFromNSString(attributionData[@"report_time"]);
            attribution.request_id = CStringFromNSString(attributionData[@"request_id"]);
            attribution.ry_touchpoint_ts = CStringFromNSString(attributionData[@"ry_touchpoint_ts"]);
            attribution.site_id = CStringFromNSString(attributionData[@"site_id"]);
            attribution.site_name = CStringFromNSString(attributionData[@"site_name"]);
            attribution.turl_campaign_id = CStringFromNSString(attributionData[@"turl_campaign_id"]);
            attribution.turl_campaign_name = CStringFromNSString(attributionData[@"turl_campaign_name"]);
            attribution.turl_campaign_name = CStringFromNSString(attributionData[@"turl_campaign_name"]);
            
            NSData *data = [NSJSONSerialization dataWithJSONObject:attributionData options:0 error:nil];
            attribution.origin_data_str = CStringFromNSString([[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding]);
        }
        
        return attribution;
    }

    // Deeplink相关
    void SolarEngineAPI::appDeeplinkOpenURL(string url)
    {
        NSString* _url = NSStringFromCString(url.c_str());
        NSLog(@"appDeeplinkOpenURL _url = %@",_url);
        NSURL* nsUrl = [NSURL URLWithString:_url];
        [[SolarEngineSDK sharedInstance] appDeeplinkOpenURL:nsUrl];
    }

    // 访客ID管理
    void SolarEngineAPI::setVisitorID(string visitorId)
    {
        NSString* _visitorId = NSStringFromCString(visitorId.c_str());
        [[SolarEngineSDK sharedInstance] setVisitorID:_visitorId];
    }

    string SolarEngineAPI::getVisitorID()
    {
        NSString* visitorId = [[SolarEngineSDK sharedInstance] visitorID];
        return visitorId ? CStringFromNSString(visitorId) : "";
    }

    // 账户ID管理
    void SolarEngineAPI::loginWithAccountID(string accountId)
    {
        NSString* _accountId = NSStringFromCString(accountId.c_str());
        [[SolarEngineSDK sharedInstance] loginWithAccountID:_accountId];
    }

    string SolarEngineAPI::getAccountID()
    {
        NSString* accountId = [[SolarEngineSDK sharedInstance] accountID];
        return accountId ? CStringFromNSString(accountId) : "";
    }

    void SolarEngineAPI::logout()
    {
        [[SolarEngineSDK sharedInstance] logout];
    }

    string SolarEngineAPI::getDistinctId()
    {
        NSString* distinctId = [[SolarEngineSDK sharedInstance] getDistinctId];
        return distinctId ? CStringFromNSString(distinctId) : "";
    }

    // 公共事件属性
    void SolarEngineAPI::setSuperProperties(const SEJSONObject& properties)
    {
        NSDictionary* _properties = NSDictionaryFromJSONObject(properties);
        [[SolarEngineSDK sharedInstance] setSuperProperties:_properties];
    }

    void SolarEngineAPI::unsetSuperProperty(string key)
    {
        NSString* _key = NSStringFromCString(key.c_str());
        [[SolarEngineSDK sharedInstance] unsetSuperProperty:_key];
    }

    void SolarEngineAPI::clearSuperProperties()
    {
        [[SolarEngineSDK sharedInstance] clearSuperProperties];
    }

    void SolarEngineAPI::setGaid(string gaid) {
        // only android
    }
    void SolarEngineAPI::setChannel(string channel){
        // only android
    }
    void SolarEngineAPI::requestTrackingAuthorization(std::function<void(int)> completionHandler) {
        [[SolarEngineSDK sharedInstance] requestTrackingAuthorizationWithCompletionHandler:^(NSUInteger status) {
            if (completionHandler) {
                completionHandler((int)status);
            }
        }];
    }


    // 用户属性管理
    void SolarEngineAPI::userInit(const SEJSONObject& properties)
    {
        NSDictionary* _properties = NSDictionaryFromJSONObject(properties);
        [[SolarEngineSDK sharedInstance] userInit:_properties];
    }

    void SolarEngineAPI::userUpdate(const SEJSONObject& properties)
    {
        NSDictionary* _properties = NSDictionaryFromJSONObject(properties);
        [[SolarEngineSDK sharedInstance] userUpdate:_properties];
    }

    void SolarEngineAPI::userAdd(const SEJSONObject& properties)
    {
        NSDictionary* _properties = NSDictionaryFromJSONObject(properties);
        [[SolarEngineSDK sharedInstance] userAdd:_properties];
    }

    void SolarEngineAPI::userUnset(const vector<string>& keys)
    {
        NSMutableArray *nsKeys = [[NSMutableArray alloc] init];
        for (const auto& key : keys) {
            [nsKeys addObject:NSStringFromCString(key.c_str())];
        }
        [[SolarEngineSDK sharedInstance] userUnset:nsKeys];
    }

    void SolarEngineAPI::userAppend(const SEJSONObject& properties)
    {
        NSDictionary* _properties = NSDictionaryFromJSONObject(properties);
        [[SolarEngineSDK sharedInstance] userAppend:_properties];
    }

    void SolarEngineAPI::userDelete(SECDeleteType deleteType)
    {
        if (deleteType == byAccountId) {
            [[SolarEngineSDK sharedInstance] userDelete:SEUserDeleteTypeByAccountId];
        } else if (deleteType == byVisitorId) {
            [[SolarEngineSDK sharedInstance] userDelete:SEUserDeleteTypeByVisitorId];
        }
    }

    // 工具方法
    void SolarEngineAPI::reportEventImmediately()
    {
        [[SolarEngineSDK sharedInstance] reportEventImmediately];
    }

    void SolarEngineAPI::trackAppReEngagement(const SEJSONObject& customProperties)
    {
        NSDictionary* _properties = NSDictionaryFromJSONObject(customProperties);
        [[SolarEngineSDK sharedInstance] trackAppReEngagement:_properties];
    }

    // 远程配置相关API实现
    void SolarEngineAPI::setDefaultRemoteConfig(const vector<SECRemoteConfigItem> &defaultConfig)
    {
        NSMutableArray *configArray = [[NSMutableArray alloc] init];
        
        for (const auto& item : defaultConfig) {
            NSMutableDictionary *configItem = [[NSMutableDictionary alloc] init];
            configItem[@"name"] = NSStringFromCString(item.name.c_str());
            configItem[@"type"] = @(item.type);
            
            
            if (item.type == 1) {
                configItem[@"value"] = NSStringFromCString(item.value.c_str());
            } else if (item.type == 2) {
                // item.value 转成Int
                configItem[@"value"] = @([NSStringFromCString(item.value.c_str()) intValue]);
            } else if (item.type == 3) {
                // item.value 转成BOOL
                NSString *boolStr = NSStringFromCString(item.value.c_str());
                BOOL boolValue = [boolStr.lowercaseString isEqualToString:@"true"] || [boolStr isEqualToString:@"1"];
                configItem[@"value"] = @(boolValue);
            } else if (item.type == 4) {
                // item.value 转成NSDictionary 或 NSArray
                NSString *jsonStr = NSStringFromCString(item.value.c_str());
                NSData *jsonData = [jsonStr dataUsingEncoding:NSUTF8StringEncoding];
                if (jsonData) {
                    id jsonObject = [NSJSONSerialization JSONObjectWithData:jsonData options:0 error:nil];
                    if (jsonObject) {
                        configItem[@"value"] = jsonObject;
                    } else {
                        // 如果JSON解析失败，作为字符串处理
                        configItem[@"value"] = jsonStr;
                    }
                } else {
                    // 如果编码失败，作为字符串处理
                    configItem[@"value"] = jsonStr;
                }
            }
            
            [configArray addObject:configItem];
        }
        
        [[SESDKRemoteConfig sharedInstance] setDefaultConfig:configArray];
    }

    void SolarEngineAPI::setRemoteConfigEventProperties(const SEJSONObject &properties)
    {
        NSDictionary *nsProperties = NSDictionaryFromJSONObject(properties);
        [[SESDKRemoteConfig sharedInstance] setRemoteConfigEventProperties:nsProperties];
    }

    void SolarEngineAPI::setRemoteConfigUserProperties(const SEJSONObject &properties)
    {
        NSDictionary *nsProperties = NSDictionaryFromJSONObject(properties);
        [[SESDKRemoteConfig sharedInstance] setRemoteConfigUserProperties:nsProperties];
    }

    string SolarEngineAPI::fastFetchRemoteConfig(const string &key)
    {
        NSString *nsKey = NSStringFromCString(key.c_str());
        id result = [[SESDKRemoteConfig sharedInstance] fastFetchRemoteConfig:nsKey];
        
        if (result) {
            if ([result isKindOfClass:[NSString class]]) {
                return CStringFromNSString((NSString *)result);
            } else if ([result isKindOfClass:[NSNumber class]]) {
                return CStringFromNSString([NSString stringWithFormat:@"%@",result]);
            } else if ([result isKindOfClass:[NSDictionary class]] || [result isKindOfClass:[NSArray class]]) {
                // 对于复杂的嵌套JSON结构，先转换为JSON字符串，再重新解析
                SEJSONObject jsonObj;
                
                // 转换为JSON字符串
                NSData *jsonData = [NSJSONSerialization dataWithJSONObject:result options:0 error:nil];
                if (jsonData) {
                    NSString *jsonString = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
                    if (jsonString) {
                        return CStringFromNSString((NSString *)jsonString);
                    }
                }
                return CStringFromNSString([NSString stringWithFormat:@"%@",result]);
            }
        }
        
        return "";
    }

    SEJSONObject SolarEngineAPI::fastFetchAllRemoteConfig()
    {
        NSDictionary *allConfig = [[SESDKRemoteConfig sharedInstance] fastFetchRemoteConfig];
        if (allConfig) {
            return JSONObjectFromNSDictionary(allConfig);
        }
        return SEJSONObject();
    }

    void SolarEngineAPI::asyncFetchRemoteConfig(const string &key, std::function<void(const string &)> completionHandler)
    {
        NSString *nsKey = NSStringFromCString(key.c_str());
        
        [[SESDKRemoteConfig sharedInstance] asyncFetchRemoteConfig:nsKey completionHandler:^(id _Nullable data) {
            string result;
            if (data) {
                if ([data isKindOfClass:[NSString class]]) {
                    result = CStringFromNSString((NSString *)data);
                } else if ([data isKindOfClass:[NSNumber class]]) {
                    result = CStringFromNSString([NSString stringWithFormat:@"%@",data]);
                } else if ([data isKindOfClass:[NSDictionary class]] || [data isKindOfClass:[NSArray class]]) {
                    // 对于复杂的嵌套JSON结构，先转换为JSON字符串，再重新解析
                    
                    // 转换为JSON字符串
                    NSData *jsonData = [NSJSONSerialization dataWithJSONObject:data options:0 error:nil];
                    if (jsonData) {
                        NSString *jsonString = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
                        result = CStringFromNSString(jsonString);
                    } else {
                        result = CStringFromNSString([NSString stringWithFormat:@"%@",data]);
                    }
                }
            }
            
            if (completionHandler) {
                completionHandler(result);
            }
        }];
    }

    void SolarEngineAPI::asyncFetchAllRemoteConfig(std::function<void(const SEJSONObject &)> completionHandler)
    {
        [[SESDKRemoteConfig sharedInstance] asyncFetchRemoteConfigWithCompletionHandler:^(NSDictionary *dict) {
            SEJSONObject result = JSONObjectFromNSDictionary(dict);
            
            if (completionHandler) {
                completionHandler(result);
            }
        }];
    }
}


#endif
