#include "SEAnalyticsJNI.h"
#include "Android/AndroidJNI.h"
#include "Android/AndroidApplication.h"
#include "Serialization/JsonWriter.h"
#include "Serialization/JsonSerializer.h"
#include "JsonObjectConverter.h"
#include "Common/SEUtil.h"

namespace solarengineanalytics
{
	static FAttributionCallback privateAttributionCallback;
	static FInitCompletedCallback privateInitCompletedCallback;
	static FonFetchRemoteConfigCallback privateOnFetchRemoteConfigCallback;
	static FonFetchAllRemoteConfigCallback privateOnFetchAllRemoteConfigCallback;
	static FDeeplinkCallBack privateDeeplinkCallBack;
	static FDeferredDeeplinkCallBack privateDeferredDeeplinkCallBack;

	static jclass GetSEClass()
	{
		static jclass SEClass = FAndroidApplication::FindJavaClassGlobalRef("com/epicgames/unreal/SEAnalyticsWrapper");
		return SEClass;
	}

	// ----------------- 公共工具方法 -----------------
	template <typename T>

	static FString ToJsonString(const T& StructData)
	{
		FString OutputString;
		FJsonObjectConverter::UStructToJsonObjectString(StructData, OutputString);
		return OutputString;
	}

	/**
	 * 公共方法：检查 JNI 调用后是否有 Java 异常
	 * @param Env JNIEnv 环境指针
	 * @param CallSite 调用位置描述（用于日志，可选）
	 */
	static void CheckJavaException(JNIEnv* Env, const char* CallSite = nullptr)
	{
		if (!Env) return;

		if (Env->ExceptionCheck())
		{
			// 打印异常堆栈（会输出到 adb logcat）
			Env->ExceptionDescribe();

			// 必须清除异常，否则后续 JNI 调用行为未定义
			Env->ExceptionClear();

			// 输出日志，提示哪个位置触发了异常
			if (CallSite)
			{
				FSELog::Error(CUR_LOG_POSITION,
				              *FString::Printf(TEXT("Java exception in %hs"), CallSite));
			}
			else
			{
				FSELog::Error(CUR_LOG_POSITION, TEXT("Java exception occurred in JNI call"));
			}
		}
	}

	static jstring ToJString(JNIEnv* Env, const FString& Str)
	{
		return Env->NewStringUTF(TCHAR_TO_UTF8(*Str));
	}

	// ----------------- JNI 方法 -----------------
	void jni_se_setDeeplinkCallback(FDeeplinkCallBack callback)
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("jni_se_setDeepLinkCallback called"));
		privateDeeplinkCallBack = callback;
	}
	void jni_se_setDeferredDeeplinkCallback(FDeferredDeeplinkCallBack callback)
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("jni_se_setDeferredDeeplinkCallback called"));
		privateDeferredDeeplinkCallBack = callback;
	}



		
	void jni_se_preInit(FString appKey)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("preInit called with appkey: %s"), *appKey));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "preInit", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, "failed to find method preInit");
			return;
		}

		jstring appKeyJ = Env->NewStringUTF(TCHAR_TO_UTF8(*appKey));
		Env->CallStaticVoidMethod(GetSEClass(), Method, appKeyJ);
		CheckJavaException(Env, "preInit");


		Env->DeleteLocalRef(appKeyJ);
	}

	// ---------- SDK 初始化 ----------
	void jni_se_init(FString appKey, FSEConfig config, FSECRemoteConfig remoteConfig)
	{
		FString ConfigJson = ToJsonString(config);
		FString RemoteConfigJson = FSEUtil::FSERConfigToJsonString(remoteConfig);
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(
				TEXT("init  called with appkey: %s config: %s"), *appKey, *ConfigJson, *RemoteConfigJson)
		);
		if (config.attributionCallback.IsBound())
		{
			privateAttributionCallback = config.attributionCallback;
		}
		if (config.initCompletedCallback.IsBound())
		{
			privateInitCompletedCallback = config.initCompletedCallback;
		}
		if (GetSEClass() == nullptr)
		{
			return;
		}
		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->
			GetStaticMethodID(GetSEClass(), "initialize", "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method initialize");
			return;
		}
		jstring appKeyJ = Env->NewStringUTF(TCHAR_TO_UTF8(*appKey));
		jstring configJsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*ConfigJson));
		jstring remoteConfigJsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*RemoteConfigJson));
		Env->CallStaticVoidMethod(GetSEClass(), Method, appKeyJ, configJsonJ, remoteConfigJsonJ);

		CheckJavaException(Env, "init");

		Env->DeleteLocalRef(appKeyJ);
		Env->DeleteLocalRef(configJsonJ);
		Env->DeleteLocalRef(remoteConfigJsonJ);
	}


	// ---------- Visitor ID ----------
	void jni_se_setVisitorID(FString visitorId)
	{
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("setVisitorID called with visitorId: %s"), *visitorId));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setVisitorID", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method se_setVisitorID");
			return;
		}

		jstring visitorIdJ = Env->NewStringUTF(TCHAR_TO_UTF8(*visitorId));
		Env->CallStaticVoidMethod(GetSEClass(), Method, visitorIdJ);
		// ✅ 检查 Java 异常
		CheckJavaException(Env, "setVisitorID");

		Env->DeleteLocalRef(visitorIdJ);
	}

	FString jni_se_getVisitorID()
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("getVisitorID called"));

		if (GetSEClass() == nullptr)
			return FString();

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "getVisitorID", "()Ljava/lang/String;");
		// ✅ 检查 Java 异常
		CheckJavaException(Env, "getVisitorID");


		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method getVisitorID");
			return FString();
		}

		jstring resultJ = (jstring)Env->CallStaticObjectMethod(GetSEClass(), Method);
		if (resultJ == nullptr)
		{
			return FString();
		}

		const char* nativeStr = Env->GetStringUTFChars(resultJ, 0);

		FString Result = FString(nativeStr);
		Env->ReleaseStringUTFChars(resultJ, nativeStr);

		Env->DeleteLocalRef(resultJ);

		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("jni_se_getVisitorID return visitorId: %s"), *Result));
		return Result;
	}

	FString jni_se_getDistinctId()
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("jni_se_getDistinctId called"));

		if (GetSEClass() == nullptr)
			return FString();

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "getDistinctId", "()Ljava/lang/String;");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method getDistinctId");
			return FString();
		}

		jstring distinctIdJ = (jstring)Env->CallStaticObjectMethod(GetSEClass(), Method);
		CheckJavaException(Env, "getDistinctId");
		if (distinctIdJ == nullptr)
		{
			FSELog::Warning(CUR_LOG_POSITION, "getDistinctId return null");
			return FString();
		}

		const char* nativeName = Env->GetStringUTFChars(distinctIdJ, nullptr);
		FString ResultName = UTF8_TO_TCHAR(nativeName);
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("getDistinctId return distinctId: %s"), *ResultName));

		Env->ReleaseStringUTFChars(distinctIdJ, nativeName);
		Env->DeleteLocalRef(distinctIdJ);

		return ResultName;
	}
	FString jni_se_getAccountId()
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("jni_se_getAccountId called"));
		if (GetSEClass() == nullptr)
			return FString();
		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "getAccountId", "()Ljava/lang/String;");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method getAccountId");
			return FString();
		}
		jstring accountIdJ = (jstring)Env->CallStaticObjectMethod(GetSEClass(), Method);
		CheckJavaException(Env, "getAccountId");
		if (accountIdJ == nullptr)
		{
			FSELog::Warning(CUR_LOG_POSITION, "getAccountId return null");
			return FString();
		}
		const char* nativeName = Env->GetStringUTFChars(accountIdJ, nullptr);
		FString ResultName = UTF8_TO_TCHAR(nativeName);
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("getAccountId return accountId: %s"), *ResultName));
			Env->ReleaseStringUTFChars(accountIdJ, nativeName);
			Env->DeleteLocalRef(accountIdJ);
			return ResultName;
	}

	void jni_se_loginWithAccountID(FString accountId)
	{
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("loginWithAccountID called with accountId: %s"), *accountId));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "loginWithAccountID", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method loginWithAccountID");
			return;
		}

		jstring accountIdJ = Env->NewStringUTF(TCHAR_TO_UTF8(*accountId));
		Env->CallStaticVoidMethod(GetSEClass(), Method, accountIdJ);
		CheckJavaException(Env, "loginWithAccountID");

		Env->DeleteLocalRef(accountIdJ);
	}

	void jni_se_logout()
	{
		FSELog::Info(CUR_LOG_POSITION, "logout called");

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "logout", "()V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method logout");
			return;
		}

		Env->CallStaticVoidMethod(GetSEClass(), Method);
		CheckJavaException(Env, "logout");
	}


	void jni_se_setChannel(FString channel)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("setChannel called with channel: %s"), *channel));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setChannel", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method setChannel");
			return;
		}

		jstring channelJ = Env->NewStringUTF(TCHAR_TO_UTF8(*channel));
		Env->CallStaticVoidMethod(GetSEClass(), Method, channelJ);
		CheckJavaException(Env, "setChannel");

		Env->DeleteLocalRef(channelJ);
	}

	void jni_se_setGaid(FString gaid)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("setGaid called with gaid: %s"), *gaid));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setGaid", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method setGaid");
			return;
		}

		jstring gaidJ = Env->NewStringUTF(TCHAR_TO_UTF8(*gaid));
		Env->CallStaticVoidMethod(GetSEClass(), Method, gaidJ);
		CheckJavaException(Env, "setGaid");

		Env->DeleteLocalRef(gaidJ);
	}

	void jni_se_setOaid(FString oaid)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("setOaid called with oaid: %s"), *oaid));

		
			if (GetSEClass() == nullptr)
				return;

			JNIEnv* Env = FAndroidApplication::GetJavaEnv();
			jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setOaid", "(Ljava/lang/String;)V");
			if (Method == nullptr)
			{
				FSELog::Error(CUR_LOG_POSITION, " failed to find method setOaid");
				return;
			}

			jstring oaidJ = Env->NewStringUTF(TCHAR_TO_UTF8(*oaid));
			Env->CallStaticVoidMethod(GetSEClass(), Method, oaidJ);
			CheckJavaException(Env, "setOaid");

			Env->DeleteLocalRef(oaidJ);
		}
		

	void jni_se_setGDPRArea(bool gdprArea)
	{
		FSELog::Info(
			CUR_LOG_POSITION, 
			FString::Printf(TEXT("setGDPRArea called with gdprArea: %d"), gdprArea)
		);

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setGDPRArea", "(Z)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, "failed to find method setGDPRArea");
			return;
		}

		Env->CallStaticVoidMethod(GetSEClass(), Method, (jboolean)gdprArea);
		CheckJavaException(Env, "setGDPRArea");

	}

	// ---------- Event Tracking ----------

	void jni_se_trackIAPWithAttributes(FSEPurchaseEventAttributes attr)
	{
		FString Json = ToJsonString(attr);
		FString customProperties = FSEUtil::ToJsonString(attr.customProperties);
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("trackIAPWithAttributes called with attr: %s, customProperties: %s"),
			                *Json, *customProperties)
		);


		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "trackIAPWithAttributes",
		                                          "(Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(
				CUR_LOG_POSITION, " failed to find method trackIAPWithAttributes");
			return;
		}

		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*Json));
		jstring customPropertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*customProperties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ, customPropertiesJ);
		CheckJavaException(Env, "trackIAPWithAttributes");

		Env->DeleteLocalRef(jsonJ);
	}

	void jni_se_trackAdClickWithAttributes(FAdClickAttributes attr)
	{
		FString Json = ToJsonString(attr);
		FString customProperties = FSEUtil::ToJsonString(attr.customProperties);

		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("trackAdClickWithAttributes called with attr: %s, customProperties: %s"),
			                *Json, *customProperties)
		);
		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "trackAdClickWithAttributes",
		                                          "(Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(
				CUR_LOG_POSITION, " failed to find method trackAdClickWithAttributes");
			return;
		}

		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*Json));
		jstring customPropertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*customProperties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ, customPropertiesJ);

		CheckJavaException(Env, "trackAdClickWithAttributes");
		Env->DeleteLocalRef(jsonJ);
		Env->DeleteLocalRef(customPropertiesJ);
	}

	void jni_se_trackAdImpressionWithAttributes(FImpressionAttributes attr)
	{
		FString Json = ToJsonString(attr);
		FString customProperties = FSEUtil::ToJsonString(attr.customProperties);

		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("trackAdImpressionWithAttributes called with attr: %s, customProperties: %s"),
			                *Json, *customProperties)
		);
		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "trackAdImpressionWithAttributes",
		                                          "(Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(
				CUR_LOG_POSITION,
				" failed to find method trackAdImpressionWithAttributes");
			return;
		}

		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*Json));
		jstring customPropertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*customProperties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ, customPropertiesJ);
		CheckJavaException(Env, "trackAdImpressionWithAttributes");
		Env->DeleteLocalRef(jsonJ);
		Env->DeleteLocalRef(customPropertiesJ);
	}

	void jni_se_trackOrderWithAttributes(FOrderEventAttributes attr)
	{
		FString Json = ToJsonString(attr);
		FString customProperties = FSEUtil::ToJsonString(attr.customProperties);

		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("trackOrderWithAttributes called with attr: %s, customProperties: %s"),
			                *Json, *customProperties)
		);

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "trackOrderWithAttributes",
		                                          "(Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method trackOrder");
			return;
		}

		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*Json));
		jstring customPropertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*customProperties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ, customPropertiesJ);
		CheckJavaException(Env, "trackOrderWithAttributes");

		Env->DeleteLocalRef(jsonJ);
		Env->DeleteLocalRef(customPropertiesJ);
	}

	void jni_se_trackRegisterWithAttributes(FRegisterEventAttributes attr)
	{
		FString Json = ToJsonString(attr);
		FString customProperties = FSEUtil::ToJsonString(attr.customProperties);

		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("trackRegisterWithAttributes called with attr: %s, customProperties: %s"),
			                *Json, *customProperties)
		);

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "trackRegisterWithAttributes",
		                                          "(Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(
				CUR_LOG_POSITION,
				" failed to find method trackRegisterWithAttributes");
			return;
		}

		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*Json));
		jstring customPropertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*customProperties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ, customPropertiesJ);

		CheckJavaException(Env, "trackRegisterWithAttributes");
		Env->DeleteLocalRef(jsonJ);
		Env->DeleteLocalRef(customPropertiesJ);
	}

	void jni_se_trackLoginWithAttributes(FLoginEventAttributes attr)
	{
		FString Json = ToJsonString(attr);
		FString customProperties = FSEUtil::ToJsonString(attr.customProperties);
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("trackLoginWithAttributes called with attr: %s, customProperties: %s"),
			                *Json, *customProperties)
		);


		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "trackLoginWithAttributes",
		                                          "(Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(
				CUR_LOG_POSITION, " failed to find method trackLoginWithAttributes");
			return;
		}

		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*Json));
		jstring customPropertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*customProperties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ, customPropertiesJ);
		CheckJavaException(Env, "trackLoginWithAttributes");
		Env->DeleteLocalRef(jsonJ);
		Env->DeleteLocalRef(customPropertiesJ);
	}

	void jni_se_trackAppAttrWithAttributes(FSEAttAttributes attr)
	{
		FString Json = ToJsonString(attr);
		FString customProperties = FSEUtil::ToJsonString(attr.customProperties);
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("jni_se_trackAppAttrWithAttributes called with attr: %s, customProperties: %s"),
			                *Json, *customProperties)
		);


		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "trackAppAttrWithAttributes",
		                                          "(Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(
				CUR_LOG_POSITION, " failed to find method trackAppAttrWithAttributes");
			return;
		}

		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*Json));
		jstring customPropertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*customProperties));

		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ, customPropertiesJ);
		CheckJavaException(Env, "trackAppAttrWithAttributes");
		Env->DeleteLocalRef(jsonJ);
		Env->DeleteLocalRef(customPropertiesJ);
	}

		void jni_se_trackAppReEngagement(FString att)
		{
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("trackAppReEngagement called with att: %s"), *att));
			if (GetSEClass() == nullptr)
				return;
		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "trackAppReEngagement",
		                                          "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method trackAppReEngagement");
			return;
		}
		jstring attJ = Env->NewStringUTF(TCHAR_TO_UTF8(*att));
		Env->CallStaticVoidMethod(GetSEClass(), Method, attJ);
		CheckJavaException(Env, "trackAppReEngagement");
		Env->DeleteLocalRef(attJ);
		
		}

	void jni_se_eventStart(FString eventName)
	{
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("eventStart called with eventName: %s"), *eventName));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "eventStart", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method eventStart");
			return;
		}

		jstring eventNameJ = Env->NewStringUTF(TCHAR_TO_UTF8(*eventName));
		Env->CallStaticVoidMethod(GetSEClass(), Method, eventNameJ);
		CheckJavaException(Env, "eventStart");
		Env->DeleteLocalRef(eventNameJ);
	}

	void jni_se_eventFinish(FString eventName, FString properties)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(
			             TEXT("eventFinish called with eventName: %s, properties: %s"), *eventName,
			             *properties));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "eventFinish",
		                                          "(Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method eventFinish");
			return;
		}

		jstring eventNameJ = Env->NewStringUTF(TCHAR_TO_UTF8(*eventName));
		jstring propertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*properties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, eventNameJ, propertiesJ);
		CheckJavaException(Env, "eventFinish");
		Env->DeleteLocalRef(eventNameJ);
		Env->DeleteLocalRef(propertiesJ);
	}

	void jni_se_track(FString eventName, FString customAttributes, FString preAttributes)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(
			             TEXT("trackEvent called with eventName: %s, customAttributes: %s, preAttributes: %s"),
			             *eventName, *customAttributes, *preAttributes));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(),
		                                          "trackEvent",
		                                          "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method trackEvent");
			return;
		}

		jstring eventNameJ = Env->NewStringUTF(TCHAR_TO_UTF8(*eventName));
		jstring customAttributesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*customAttributes));
		jstring preAttributesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*preAttributes));
		Env->CallStaticVoidMethod(GetSEClass(), Method, eventNameJ, customAttributesJ, preAttributesJ);
		CheckJavaException(Env, "trackEvent");
		Env->DeleteLocalRef(eventNameJ);
		Env->DeleteLocalRef(customAttributesJ);
		Env->DeleteLocalRef(preAttributesJ);
	}

	void jni_se_trackFirst(FString att)
	{
		// FString Json = FSEUtil::FirstEventToJsonString(att);
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("trackFirst called with att: %s"), *att));


		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "trackFirst", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method trackFirst");
			return;
		}

		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*att));

		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ);
		CheckJavaException(Env, "trackFirst");

		Env->DeleteLocalRef(jsonJ);
	}

	void jni_se_reportEventImmediately()
	{
		FSELog::Info(CUR_LOG_POSITION, "jni_se_reportEventImmediately called");

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "reportEventImmediately", "()V");
		if (Method == nullptr)
		{
			FSELog::Error(
				CUR_LOG_POSITION, "jni_se_reportEventImmediately failed to find method reportEventImmediately");
			return;
		}

		Env->CallStaticVoidMethod(GetSEClass(), Method);
	}


	// ---------- User APIs ----------

	void jni_se_userInit(FString properties)
	{
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("userInit called with properties: %s"), *properties));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "userInit", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method userInit");
			return;
		}

		jstring propertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*properties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, propertiesJ);
		CheckJavaException(Env, "userInit");
		Env->DeleteLocalRef(propertiesJ);
	}

	void jni_se_userSet(FString properties)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("userSet called with properties: %s"), *properties));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "userSet", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method userSet");
			return;
		}

		jstring propertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*properties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, propertiesJ);
		CheckJavaException(Env, "userSet");
		Env->DeleteLocalRef(propertiesJ);
	}


	void jni_se_userAdd(FString properties)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("userAdd called with properties: %s"), *properties));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "userAdd", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method userAdd");
			return;
		}

		jstring propertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*properties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, propertiesJ);
		CheckJavaException(Env, "userAdd");
		Env->DeleteLocalRef(propertiesJ);
	}

	void jni_se_userUnset(FString Key)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("userUnset called with key: %s"), *Key));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "userUnset", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method userUnset");
			return;
		}

		jstring keyJ = Env->NewStringUTF(TCHAR_TO_UTF8(*Key));
		Env->CallStaticVoidMethod(GetSEClass(), Method, keyJ);
		CheckJavaException(Env, "userUnset");
		Env->DeleteLocalRef(keyJ);
	}

	void jni_se_userAppend(FString properties)
	{
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("userAppend called with properties: %s"), *properties));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "userAppend", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method userAppend");
			return;
		}

		jstring propertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*properties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, propertiesJ);
		CheckJavaException(Env, "userAppend");
		Env->DeleteLocalRef(propertiesJ);
	}

	void jni_se_userUpdate(FString properties)
	{
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("userUpdate called with properties: %s"), *properties));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "userUpdate", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method userUpdate");
			return;
		}

		jstring propertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*properties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, propertiesJ);
		CheckJavaException(Env, "userUpdate");
		Env->DeleteLocalRef(propertiesJ);
	}

	void jni_se_userDelete(int deleteType)
	{
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("userDelete called with deleteType: %d"), deleteType));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "userDelete", "(I)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method userDelete");
			return;
		}

		Env->CallStaticVoidMethod(GetSEClass(), Method, deleteType);
		CheckJavaException(Env, "userDelete");
	}

	// ---------- SuperProperties(----------
	void jni_se_setSuperProperties(FString properties)
	{
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("setSuperProperties called with properties: %s"), *properties));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setSuperProperties", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method setSuperProperties");
			return;
		}

		jstring propertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*properties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, propertiesJ);
		CheckJavaException(Env, "setSuperProperties");
		Env->DeleteLocalRef(propertiesJ);
	}

	void jni_se_unsetSuperProperty(FString propertyName)
	{
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("unsetSuperProperty called with propertyName: %s"), *propertyName));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "unsetSuperProperty", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method unsetSuperProperty");
			return;
		}

		jstring propertyNameJ = Env->NewStringUTF(TCHAR_TO_UTF8(*propertyName));
		Env->CallStaticVoidMethod(GetSEClass(), Method, propertyNameJ);
		CheckJavaException(Env, "unsetSuperProperty");
		Env->DeleteLocalRef(propertyNameJ);
	}

	void jni_se_clearSuperProperties()
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("clearSuperProperties called"));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "clearSuperProperties", "()V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method clearSuperProperties");
			return;
		}

		Env->CallStaticVoidMethod(GetSEClass(), Method);
		CheckJavaException(Env, "clearSuperProperties");
	}

	void jni_se_setPresetEvent(ESEPresetEventType EventType, FString Properties)
	{
		FSELog::Info(CUR_LOG_POSITION,
		             FString::Printf(TEXT("setPresetEvent called with EventType: %d, properties: %s"),
		                             (int32)EventType, *Properties));

		if (GetSEClass() == nullptr)
			return;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setPresetEvent", "(ILjava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method setPresetEvent");
			return;
		}

		jstring propertiesJ = Env->NewStringUTF(TCHAR_TO_UTF8(*Properties));
		Env->CallStaticVoidMethod(GetSEClass(), Method, (int32)EventType, propertiesJ);
		CheckJavaException(Env, "setPresetEvent");
		Env->DeleteLocalRef(propertiesJ);
	}

	TSharedPtr<FJsonObject> jni_se_getPresetProperties()
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("jni_se_getPresetProperties called"));

		if (GetSEClass() == nullptr)
			return nullptr;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "getPresetProperties", "()Ljava/lang/String;");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method getPresetProperties");
			return nullptr;
		}

		jstring resultJ = (jstring)Env->CallStaticObjectMethod(GetSEClass(), Method);
		CheckJavaException(Env, "getPresetProperties");
		if (!resultJ)
			return nullptr;

		const char* UTFChars = Env->GetStringUTFChars(resultJ, nullptr);
		FString JsonStr = UTF8_TO_TCHAR(UTFChars);
		Env->ReleaseStringUTFChars(resultJ, UTFChars);
		Env->DeleteLocalRef(resultJ);

		TSharedPtr<FJsonObject> JsonObj;
		TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(JsonStr);
		if (FJsonSerializer::Deserialize(Reader, JsonObj) && JsonObj.IsValid())
		{
			return JsonObj;
		}

		return nullptr;
	}

	TSharedPtr<FJsonObject> jni_se_getAttributionData()
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("jni_se_getAttributionData called"));

		if (GetSEClass() == nullptr)
			return nullptr;

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "getAttributionData", "()Ljava/lang/String;");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method getAttributionData");
			return nullptr;
		}

		jstring resultJ = (jstring)Env->CallStaticObjectMethod(GetSEClass(), Method);
		CheckJavaException(Env, "getAttributionData");

		if (!resultJ)
			return nullptr;

		const char* UTFChars = Env->GetStringUTFChars(resultJ, nullptr);
		FString JsonStr = UTF8_TO_TCHAR(UTFChars);
		Env->ReleaseStringUTFChars(resultJ, UTFChars);
		Env->DeleteLocalRef(resultJ);

		TSharedPtr<FJsonObject> JsonObj;
		TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(JsonStr);
		if (FJsonSerializer::Deserialize(Reader, JsonObj) && JsonObj.IsValid())
		{
			return JsonObj;
		}

		return nullptr;
	}


	extern "C"
JNIEXPORT void JNICALL

	Java_com_epicgames_unreal_SEAnalyticsWrapper_onAttributionResult(
		JNIEnv* Env,
		jclass clazz,
		jint code,
		jstring jsonStr)
	{
		
		const char* utfChars = Env->GetStringUTFChars(jsonStr, nullptr);
		FString ResultJson = UTF8_TO_TCHAR(utfChars);
		Env->ReleaseStringUTFChars(jsonStr, utfChars);


		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(
				TEXT("onAttributionResult called with code: %d   AttributionData: %s"), (int32)code, *ResultJson)
		);

		// 转 FJsonObject
		TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(ResultJson);
		TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();


		if (FJsonSerializer::Deserialize(Reader, JsonObject))
		{
			FSEAttributionData Data;
			Data.code = (int32)code;
			Data.data = JsonObject;

			// 调用 UE 委托
			if (privateAttributionCallback.IsBound())
			{
				FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] Executing privateAttributionCallback...")));
				privateAttributionCallback.Execute( Data);
			}
			else
			{
				FSELog::Error(
					CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] privateAttributionCallback is not bound!")));
			}
		}
		else
		{
			FSELog::Warning(CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] Failed to parse JSON: %s"), *ResultJson));
		}
	}


	extern "C"
JNIEXPORT void JNICALL

	Java_com_epicgames_unreal_SEAnalyticsWrapper_onInitCompletedResult(
		JNIEnv* Env,
		jclass clazz,
		jint code)
	{
		
		FSELog::Info(
			CUR_LOG_POSITION, FString::Printf(TEXT("onInitCompletedResult called with code: %d"), (int32)code));

		// 调用 UE 委托
		if (privateInitCompletedCallback.IsBound())
		{
			FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] Executing privateInitCompletedCallback...")));
			privateInitCompletedCallback.Execute((int32)code);
		}
		else
		{
			FSELog::Warning(CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] onInitCompletedResult is not bound!")));
		}
	}

	extern "C"
JNIEXPORT void JNICALL

	Java_com_epicgames_unreal_SEAnalyticsWrapper_OnDeferredDeeplinkCompleted(
		JNIEnv* Env,
		jclass clazz,
		jint code,
		jstring jsonStr)
	{
		
		const char* utfChars = Env->GetStringUTFChars(jsonStr, nullptr);
		FString ResultJson = UTF8_TO_TCHAR(utfChars);
		Env->ReleaseStringUTFChars(jsonStr, utfChars);
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(
				TEXT("onDeferredDeeplinkResult called with code: %d   DeferredDeeplinkData: %s"), (int32)code, *ResultJson)
		);
		// 转 FJsonObject
		TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(ResultJson);
		TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
		if (FJsonSerializer::Deserialize(Reader, JsonObject))
		{
			FDeferredDeeplinkData Data;
			Data.code = (int32)code;
			Data.sedpLink      = JsonObject->GetStringField(TEXT("sedpLink"));
			Data.sedpUrlscheme = JsonObject->GetStringField(TEXT("sedpUrlscheme"));
			Data.turlId        = JsonObject->GetStringField(TEXT("turlId"));
			// 调用 UE 委托
			if (privateDeferredDeeplinkCallBack.IsBound())
				privateDeferredDeeplinkCallBack.Execute(Data);
				else
					FSELog::Error(
						CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] privateDeferredDeeplinkCallback is not bound!")));
		}
	
	
	}

extern "C"
JNIEXPORT void JNICALL
Java_com_epicgames_unreal_SEAnalyticsWrapper_OnDeeplinkCompleted(
		JNIEnv* Env,
		jclass clazz,
		jint code,
		jstring jsonStr)
{
		const char* utfChars = Env->GetStringUTFChars(jsonStr, nullptr);
		FString ResultJson = UTF8_TO_TCHAR(utfChars);
		Env->ReleaseStringUTFChars(jsonStr, utfChars);
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(
				TEXT("OnDeeplinkCompleted called with code: %d   DeeplinkData: %s"), (int32)code, *ResultJson)
		);
		TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(ResultJson);
		TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
		if (FJsonSerializer::Deserialize(Reader, JsonObject))
		{
			FDeeplinkData Data;
			Data.code = (int32)code;
			Data.from = JsonObject->GetStringField(TEXT("from"));
			Data.baseUrl    = JsonObject->GetStringField(TEXT("baseUrl"));
			Data.sedpLink   = JsonObject->GetStringField(TEXT("sedpLink"));
			Data.turlId     = JsonObject->GetStringField(TEXT("turlId"));
			Data.url       = JsonObject->GetStringField(TEXT("url"));
			Data.customParams = JsonObject->GetObjectField(TEXT("customParams"));
			
			FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] Executing privateDeeplinkCallback...")));
			privateDeeplinkCallBack.Execute(Data);
		}
		else
		{
			FSELog::Warning(CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] Failed to parse JSON: %s"), *ResultJson));
		}
 }















	//在线参数
	void jni_se_setRemoteDefaultConfig(FString json)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("jni_se_getRemoteConfigs called with json: %s"), *json));
		if (GetSEClass() == nullptr)
			return;
		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setRemoteDefaultConfig", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method getRemoteConfigs");
			return;
		}
		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*json));
		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ);
		CheckJavaException(Env, "setRemoteDefaultConfig");
		Env->DeleteLocalRef(jsonJ);
	}









	void jni_se_setRemoteConfigEventProperties(FString json)
	{
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("jni_se_setRemoteConfigEventProperties called with json: %s"), *json));
		if (GetSEClass() == nullptr)
			return;
		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setRemoteConfigEventProperties",
		                                          "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method setRemoteConfigEventProperties");
			return;
		}
		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*json));
		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ);
		CheckJavaException(Env, "setRemoteConfigEventProperties");
		Env->DeleteLocalRef(jsonJ);
	}

	void jni_se_setRemoteConfigUserProperties(FString json)
	{
		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("jni_se_setRemoteConfigUserProperties called with json: %s"), *json));
		if (GetSEClass() == nullptr)
			return;
		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "setRemoteConfigUserProperties",
		                                          "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method setRemoteConfigUserProperties");
			return;
		}
		jstring jsonJ = Env->NewStringUTF(TCHAR_TO_UTF8(*json));
		Env->CallStaticVoidMethod(GetSEClass(), Method, jsonJ);
		CheckJavaException(Env, "setRemoteConfigUserProperties");
		Env->DeleteLocalRef(jsonJ);
	}

	//
	// void jni_se_setRemoteDefaultConfig(FString defaultConfig);
	// void jni_se_setRemoteConfigEventProperties(FString properties);
	// void jni_se_setRemoteConfigUserProperties(FString properties);
	// void jni_se_fastFetchRemoteConfig(FString key);
	// void jni_se_fastAllFetchRemoteConfig();
	// void jni_se_asyncFetchRemoteConfig(FString key, FonFetchRemoteConfigCallback callback);
	// void jni_se_asyncAllFetchRemoteConfig(FonFetchRemoteConfigCallback callback);
	TSharedPtr<FJsonObject> jni_se_fastAllFetchRemoteConfig()
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("jni_se_fastAllFetchRemoteConfig called"));
		if (GetSEClass() == nullptr)
			return nullptr;
		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "fastAllFetchRemoteConfig", "()Ljava/lang/String;");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method fastAllFetchRemoteConfig");
			return nullptr;
		}
		jstring jsonJ = (jstring)Env->CallStaticObjectMethod(GetSEClass(), Method);
		CheckJavaException(Env, "fastAllFetchRemoteConfig");
		if (jsonJ == nullptr)
			return nullptr;
		const char* utfChars = Env->GetStringUTFChars(jsonJ, nullptr);
		FString ResultJson = UTF8_TO_TCHAR(utfChars);
		Env->ReleaseStringUTFChars(jsonJ, utfChars);
		Env->DeleteLocalRef(jsonJ);
		TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(ResultJson);
		TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
		if (FJsonSerializer::Deserialize(Reader, JsonObject))
		{
			return JsonObject;
		}
		return nullptr;
	}

	FString jni_se_fastFetchRemoteConfig(FString key)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("fastFetchRemoteConfig called with key: %s"), *key));

		if (!GetSEClass())
			return FString();

		JNIEnv* Env = FAndroidApplication::GetJavaEnv();

		jmethodID Method = Env->GetStaticMethodID(
			GetSEClass(),
			"fastFetchRemoteConfig",
			"(Ljava/lang/String;)Ljava/lang/String;"
		);

		if (!Method)
		{
			FSELog::Error(CUR_LOG_POSITION, "failed to find method fastFetchRemoteConfig");
			return FString();
		}
		jstring keyJ = Env->NewStringUTF(TCHAR_TO_UTF8(*key));
		jstring jsonJ = (jstring)Env->CallStaticObjectMethod(GetSEClass(), Method, keyJ);
		CheckJavaException(Env, "fastFetchRemoteConfig");
		if (jsonJ == nullptr)
			return FString();
		const char* utfChars = Env->GetStringUTFChars(jsonJ, nullptr);
		FString ResultJson = UTF8_TO_TCHAR(utfChars);
		Env->ReleaseStringUTFChars(jsonJ, utfChars);
		Env->DeleteLocalRef(jsonJ);
		return ResultJson;
	}

	void jni_se_asyncAllFetchRemoteConfig(FonFetchAllRemoteConfigCallback callback)
	{
		FSELog::Info(CUR_LOG_POSITION, TEXT("jni_se_asyncAllFetchRemoteConfig called"));
		if (GetSEClass() == nullptr)
			return;
		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "asyncAllFetchRemoteConfig", "()V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method asyncAllFetchRemoteConfig");
			return;
		}
		Env->CallStaticVoidMethod(GetSEClass(), Method);
		if (callback.IsBound())
			privateOnFetchAllRemoteConfigCallback = callback;
		CheckJavaException(Env, "asyncAllFetchRemoteConfig");
	}

	void jni_se_asyncFetchRemoteConfig(FString key, FonFetchRemoteConfigCallback callback)
	{
		FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("asyncFetchRemoteConfig called with key: %s"), *key));
		if (GetSEClass() == nullptr)
			return;
		JNIEnv* Env = FAndroidApplication::GetJavaEnv();
		jmethodID Method = Env->GetStaticMethodID(GetSEClass(), "asyncFetchRemoteConfig", "(Ljava/lang/String;)V");
		if (Method == nullptr)
		{
			FSELog::Error(CUR_LOG_POSITION, " failed to find method asyncFetchRemoteConfig");
			return;
		}
		jstring keyJ = Env->NewStringUTF(TCHAR_TO_UTF8(*key));
		Env->CallStaticVoidMethod(GetSEClass(), Method, keyJ);
		if (callback.IsBound())
			privateOnFetchRemoteConfigCallback = callback;
		CheckJavaException(Env, "asyncFetchRemoteConfig");
		Env->DeleteLocalRef(keyJ);
	}


	extern "C"
JNIEXPORT void JNICALL

	Java_com_epicgames_unreal_SEAnalyticsWrapper_OnRemoteConfigReceivedData(JNIEnv* env, jclass clazz, jstring value)
	{
		if (privateOnFetchRemoteConfigCallback.IsBound())
		{
			FSELog::Info(
				CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] Executing privateOnFetchRemoteConfigCallback...")));

			FString valueStr = FJavaHelper::FStringFromParam(env, value);
			FSELog::Info(CUR_LOG_POSITION, FString::Printf(
				             TEXT("[JNI] privateOnFetchRemoteConfigCallback executed with value: %s"), *valueStr));
			privateOnFetchRemoteConfigCallback.Execute(valueStr);
		}
		else
			FSELog::Warning(CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] OnRemoteConfigReceivedData is not bound!")));


		return;
	}

	extern "C"
	JNIEXPORT void JNICALL

	Java_com_epicgames_unreal_SEAnalyticsWrapper_OnRemoteConfigReceivedAllData(
		JNIEnv* env, jclass clazz, jstring jsonStr)
	{
		const char* utfChars = env->GetStringUTFChars(jsonStr, nullptr);
		FString ResultJson = UTF8_TO_TCHAR(utfChars);
		env->ReleaseStringUTFChars(jsonStr, utfChars);

		FSELog::Info(
			CUR_LOG_POSITION,
			FString::Printf(TEXT("OnRemoteConfigReceivedAllData called with value: %s"), *ResultJson));
		TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(ResultJson);
		TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
		if (FJsonSerializer::Deserialize(Reader, JsonObject))
		{
			FSERemoteConfigData data;
			data.data = JsonObject;
			if (privateOnFetchAllRemoteConfigCallback.IsBound())
			{
				FSELog::Info(
					CUR_LOG_POSITION,
					FString::Printf(TEXT("[JNI] Executing privateOnFetchAllRemoteConfigCallback...")));
				privateOnFetchAllRemoteConfigCallback.Execute(data);
			}
			else
				FSELog::Warning(
					CUR_LOG_POSITION, FString::Printf(TEXT("[JNI] OnRemoteConfigReceivedAllData is not bound!")));
		}
		else
		{
			FSELog::Error(
				CUR_LOG_POSITION,
				FString::Printf(TEXT("[JNI] OnRemoteConfigReceivedAllData failed to parse json: %s"), *ResultJson));
		}
	}
}
