//// Copyright 2020 Thinking Data Ltd. All Rights Reserved.
//

#if PLATFORM_IOS
#import <SolarEngineSDK/SolarEngineSDK.h>
#import <SESDKRemoteConfig/SESDKRemoteConfig.h>

#if  __has_include(<SolarEngineSDK/SESDKForCN.h>)
#import <SolarEngineSDK/SESDKForCN.h>
#endif
#if  __has_include(<SolarEngineSDK/SESDKForUS.h>)
#import <SolarEngineSDK/SESDKForUS.h>
#endif
#import "SEAnalyticsCpp.h"

NSMutableDictionary* sInstances;
NSMutableArray*      sAppIds;


inline NSString* FStringToNSString(const FString& str)
{
    return   (*str != nullptr && !str.IsEmpty()) ? [NSString stringWithUTF8String:TCHAR_TO_UTF8(*str)] : nil;
}
inline FString NSStringToFString(NSString* nsString)
{
    if (nsString == nil)
    {
        return FString("");
    }
    const char* utf8Str = [nsString UTF8String];
    return FString(UTF8_TO_TCHAR(utf8Str));
}
const TCHAR* SafeStr(const FString& Str)
{
   return Str.IsEmpty() ? TEXT("") : *Str;
}


NSDictionary* convertToDictionary(const char *json)
{
    if (json == NULL) return nil;

    NSString *jsonString = [NSString stringWithUTF8String:json];
    if (!jsonString) return nil;

    NSData *data = [jsonString dataUsingEncoding:NSUTF8StringEncoding];
    if (!data) return nil;

    NSDictionary *dict = [NSJSONSerialization JSONObjectWithData:data options:kNilOptions error:nil];
    return dict;
}

// 将 NSDictionary 转换为 UE4 JsonObject
TSharedPtr<FJsonObject> convertNSDictionaryToJsonObject(NSDictionary* Dict)
{
    if (!Dict) return MakeShared<FJsonObject>();

    TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();

    for (id Key in Dict)
    {
        id Value = Dict[Key];

        // Key 转 FString
        FString KeyStr = UTF8_TO_TCHAR([[Key description] UTF8String]);

        // Value 转 FString
        FString ValueStr = UTF8_TO_TCHAR([[Value description] UTF8String]);

        JsonObject->SetStringField(KeyStr, ValueStr);
    }

    return JsonObject;
}

NSDictionary* convertFJsonObjectToNSDictionary(TSharedPtr<FJsonObject> JsonObject)
{
    if (!JsonObject.IsValid()) return nil;

    // 1. 序列化 UE 的 FJsonObject 到 FString
    FString OutputString;
    TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&OutputString);
    if (!FJsonSerializer::Serialize(JsonObject.ToSharedRef(), Writer))
    {
        // 序列化失败
        return nil;
    }

    // 2. 转成 NSDictionary*
    return convertToDictionary(TCHAR_TO_UTF8(*OutputString));
}

const char *convertToString(NSDictionary *properties_dict)
{

    if ( properties_dict && [NSJSONSerialization isValidJSONObject:properties_dict] )
    {
        NSData *jsonData = [NSJSONSerialization dataWithJSONObject:properties_dict options:0 error:NULL];;
        NSString *output = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
        if ( output )
        {
            return [output UTF8String];
        }
        else
        {
            return "";
        }
    }
    else
    {
        return "";
    }
}
//
NSMutableDictionary* getsInstances()
{
    if( sInstances == nil )
    {
        sInstances = [NSMutableDictionary new];
    }
    return  sInstances;
}

@interface SEWrapperManager : NSObject

@property (nonatomic,copy)NSString *sub_lib_version;
@property (nonatomic,copy)NSString *sdk_type;

+(SEWrapperManager *)sharedInstance;

@end
@implementation SEWrapperManager

+ (SEWrapperManager *)sharedInstance {
    static SEWrapperManager * instance = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        instance = [[self alloc] init];
    });
    return instance;
}

- (instancetype)init
{
    self = [super init];
    if (self) {
        _sdk_type = @"unreal";
    }
    return self;
}

@end



FDeferredDeeplinkCallBack se_deferredDeeplinkCallback;
FDeeplinkCallBack se_deeplinkCallback;
FAttributionCallback se_attributionCallback;
FInitCompletedCallback se_initCompletedCallback;

FonFetchRemoteConfigCallback se_onFetchRemoteConfigCallback;
FonFetchAllRemoteConfigCallback se_onFetchAllRemoteConfigCallback;
FonRequestTrackingAuthorizationCallback se_onRequestTrackingAuthorizationCallback;



void SEAnalyticsCpp::se_preInit(FString appkey){
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[preInit] called, appkey: %s"), *appkey));
    NSString *appkey_string = FStringToNSString(appkey);
    [[SolarEngineSDK sharedInstance] preInitWithAppKey:appkey_string];
}



void SEAnalyticsCpp::se_init(FString appkey, FSEConfig config,FSECRemoteConfig rcConfig){
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[init] called, appkey: %s"), *appkey));
    SEConfig *seConfig = [[SEConfig alloc] init];
    if(config.attributionCallback.IsBound())
        SEAnalyticsCpp::se_setAttributionCallback(config.attributionCallback);
    if(config.initCompletedCallback.IsBound())
        SEAnalyticsCpp:: se_setInitCompletedCallback(config.initCompletedCallback);
    
    // 使用传入的配置参数
    seConfig.isGDPRArea = config.isGDPRArea;
    seConfig.logEnabled = config.logEnabled;
    seConfig.isDebugModel = config.isDebugModel;
    seConfig.setCoppaEnabled = config.isCoppaEnabled;
    seConfig.setKidsAppEnabled = config.isKidsAppEnabled;
    seConfig.attAuthorizationWaitingInterval = config.attAuthorizationWaitingInterval;
    seConfig.enableDeferredDeeplink = config.deferredDeeplinkenable;
    
    seConfig.enable2GReporting= config.enable2GReporting;
#if  __has_include(<SolarEngineSDK/SESDKForCN.h>)
    seConfig.caid=FStringToNSString(config.caid);
#endif
#if  __has_include(<SolarEngineSDK/SESDKForUS.h>)
    seConfig.enableODMInfo = config.enableODMInfo;
#endif
   
    if(config.customDomain.enable){
        SECustomDomain *customDomain = [[SECustomDomain alloc] init];
        customDomain.enable = config.customDomain.enable;
        customDomain.receiverDomain = FStringToNSString(config.customDomain.receiverDomain);
        customDomain.ruleDomain = FStringToNSString(config.customDomain.ruleDomain);
        customDomain.receiverTcpHost = FStringToNSString(config.customDomain.ruleTcpHost);
        customDomain.gatewayTcpHost = FStringToNSString(config.customDomain.gatewayTcpHost);
        seConfig.customDomain = customDomain;
    }
    if(rcConfig.enable){
        SERemoteConfig *remoteConfig = [[SERemoteConfig alloc] init];
        remoteConfig.enable = rcConfig.enable;
        remoteConfig.logEnabled = config.logEnabled;
        remoteConfig.mergeType = (SERCMergeType)rcConfig.mergeType;
        remoteConfig.customIDProperties = convertFJsonObjectToNSDictionary(rcConfig.customIDProperties);
        remoteConfig.customIDEventProperties =convertFJsonObjectToNSDictionary( rcConfig.customIDEventProperties);
        remoteConfig.customIDUserProperties =convertFJsonObjectToNSDictionary( rcConfig.customIDUserProperties);
        seConfig.remoteConfig = remoteConfig;
    }
    [SEWrapperManager sharedInstance].sub_lib_version = FStringToNSString(SE_UE_LIB_VERSION);
    
    
    
    [[SolarEngineSDK sharedInstance] startWithAppKey:FStringToNSString(appkey) config:seConfig];
}
//

//
FString SEAnalyticsCpp::se_getDistinctId()
{
    FSELog::Info(CUR_LOG_POSITION, TEXT("[getDistinctId] called"));
    NSString *distinctId =  [[SolarEngineSDK sharedInstance] getDistinctId];
    const char *str = [distinctId cStringUsingEncoding:NSUTF8StringEncoding];
    FString str1(str);
    return str1;
}

void SEAnalyticsCpp::se_setGDPRArea( bool isGDPRArea)
{ 
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[setGDPRArea] called, isGDPRArea: %d"), isGDPRArea));
    [[SolarEngineSDK sharedInstance] setGDPRArea:isGDPRArea];
}

FString SEAnalyticsCpp::se_getVisitorID()
{
    FSELog::Info(CUR_LOG_POSITION, TEXT("[getVisitorID] called"));
    NSString *visitor = [[SolarEngineSDK sharedInstance] visitorID];
    if (!visitor) { return FString(""); }
    const char *str = [visitor cStringUsingEncoding:NSUTF8StringEncoding];
    return FString(str);
}

FString SEAnalyticsCpp::se_getAccountId() {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[getAccountId] called"));
    NSString *accountId =[[SolarEngineSDK sharedInstance] accountID];
    if (!accountId) { return FString(""); }
    const char *str = [accountId cStringUsingEncoding:NSUTF8StringEncoding];
    return FString(str);
    
}

void SEAnalyticsCpp::se_track(FString eventName, FString customAttributes, FString preAttributes) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[track] called, eventName: %s, customAttributes: %s, preAttributes: %s"), *eventName, *customAttributes, *preAttributes));
    [[SolarEngineSDK sharedInstance] track:FStringToNSString(eventName)
                      withCustomProperties:convertToDictionary(TCHAR_TO_UTF8(*customAttributes))
                      withPresetProperties:convertToDictionary(TCHAR_TO_UTF8(*preAttributes))];
}

void SEAnalyticsCpp::se_trackFirst(FString att)
{
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[trackFirst] called, att: %s"), *att));
    NSDictionary *dict =convertToDictionary(TCHAR_TO_UTF8(*att));
    NSString *eventType = [dict objectForKey:@"eventType"];
    NSString *checkId  =  [dict objectForKey:@"checkId"];
    NSDictionary *customProperties = [dict objectForKey:@"customProperties"];

    if ([eventType isEqualToString:@"_appReg"]) {
        SERegisterEventAttribute *eventAttribute = [[SERegisterEventAttribute alloc] init];
        eventAttribute.registerType   = [dict objectForKey:@"registerType"];
        eventAttribute.registerStatus = [dict objectForKey:@"registerStatus"];
        eventAttribute.firstCheckId =checkId;
        eventAttribute.customProperties = customProperties;
        [[SolarEngineSDK sharedInstance] trackFirstEvent:eventAttribute];
    }
    else if ([eventType isEqualToString:@"_customEvent"]) {
        SECustomEventAttribute *customAttribute = [[SECustomEventAttribute alloc] init];
        customAttribute.firstCheckId =checkId;
        customAttribute.eventName= [dict objectForKey:@"customEventName"];
        customAttribute.customProperties = customProperties;
        [[SolarEngineSDK sharedInstance] trackFirstEvent:customAttribute];
    }
    else {
        NSLog(@"[SE] Unknown eventType: %@", eventType);
    }
}



void SEAnalyticsCpp::se_userInit(FString properties)
{
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[userInit] called, properties: %s"), *properties));
    NSDictionary *properties_dict =  convertToDictionary(TCHAR_TO_UTF8(*properties));
    [[SolarEngineSDK sharedInstance] userInit:properties_dict];
}

void SEAnalyticsCpp::se_userUpdate(FString properties)
{
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[serUpdate] called, properties: %s"), *properties));
    NSDictionary *dict = convertToDictionary(TCHAR_TO_UTF8(*properties));
    if (dict)
        [[SolarEngineSDK sharedInstance] userUpdate:dict];
   
}
void SEAnalyticsCpp::se_userAdd(FString properties)
    {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[userAdd] called, properties: %s"), *properties));
    NSDictionary *properties_dict =  convertToDictionary(TCHAR_TO_UTF8(*properties));
    [[SolarEngineSDK sharedInstance] userAdd:properties_dict];
}

void SEAnalyticsCpp::se_userUnset(TArray<FString> keys)
    {
    FString KeysStr = FString::Join(keys, TEXT(","));
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[userUnset] called, keys: %s"), *KeysStr));
    NSMutableArray<NSString *> *KeysArray = [NSMutableArray arrayWithCapacity:keys.Num()];
    for (const FString& Key : keys)
    {
        NSString *KeyStr = [NSString stringWithUTF8String:TCHAR_TO_UTF8(*Key)];
        [KeysArray addObject:KeyStr];
    }
    // 调用 SDK 的 userUnset:
    [[SolarEngineSDK sharedInstance] userUnset:KeysArray];
}

void SEAnalyticsCpp::se_userAppend(FString properties) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[userAppend] called, properties: %s"), *properties));
    NSDictionary *properties_dict =  convertToDictionary(TCHAR_TO_UTF8(*properties));
    [[SolarEngineSDK sharedInstance] userAppend:properties_dict];
}

void SEAnalyticsCpp::se_userDelete(int deleteType) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[userDelete] called, deleteType: %d"), deleteType));
    [[SolarEngineSDK sharedInstance] userDelete:deleteType];
}

void SEAnalyticsCpp::se_setSuperProperties(FString properties) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[setSuperProperties] called, properties: %s"), *properties));
    NSDictionary *properties_dict =  convertToDictionary(TCHAR_TO_UTF8(*properties));
    [[SolarEngineSDK sharedInstance] setSuperProperties:properties_dict];
}

void SEAnalyticsCpp::se_unsetSuperProperty(FString property) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[unsetSuperProperty] called, property: %s"), *property));
    NSString *propertyStr = [NSString stringWithUTF8String:TCHAR_TO_UTF8(*property)];
    [[SolarEngineSDK sharedInstance] unsetSuperProperty:propertyStr];
}

void SEAnalyticsCpp::se_setPresetEvent(ESEPresetEventType EventType, FString properties) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[setPresetEvent] called, EventType: %d, properties: %s"), (int)EventType, *properties));
    NSDictionary *properties_dict = convertToDictionary(TCHAR_TO_UTF8(*properties));
    if (!properties_dict) {
        FSELog::Warning(CUR_LOG_POSITION, TEXT("[setPresetEvent] received invalid JSON properties"));
    }
    switch (EventType) {
        case ESEPresetEventType::AppInstall:
            [[SolarEngineSDK sharedInstance] setPresetEvent:SEPresetEventTypeAppInstall
                                             withProperties:properties_dict];
            break;
        case ESEPresetEventType::AppStart:
            [[SolarEngineSDK sharedInstance] setPresetEvent:SEPresetEventTypeAppStart
                                             withProperties:properties_dict];
            break;
        case ESEPresetEventType::AppEnd:
            [[SolarEngineSDK sharedInstance] setPresetEvent:SEPresetEventTypeAppEnd
                                             withProperties:properties_dict];
            break;
        case ESEPresetEventType::All:
            [[SolarEngineSDK sharedInstance] setPresetEvent:SEPresetEventTypeAppAll
                                             withProperties:properties_dict];
            break;
        default:
            FSELog::Error(CUR_LOG_POSITION,
                FString::Printf(TEXT("[setPresetEvent] Unknown preset event type: %d"), (int)EventType));
            break;
    }
}





void SEAnalyticsCpp::se_loginWithAccountID(FString accountId) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[loginWithAccountID] called, accountId: %s"), *accountId));
    [[SolarEngineSDK sharedInstance] loginWithAccountID:FStringToNSString(accountId)];
}

void SEAnalyticsCpp::se_logout() {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[logout] called"));
    [[SolarEngineSDK sharedInstance] logout];
}
void SEAnalyticsCpp::se_setVisitorID(FString visitorId) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[setVisitorID] called, visitorId: %s"), *visitorId));
    [[SolarEngineSDK sharedInstance] setVisitorID:FStringToNSString(visitorId)];
}


id seTrimValue(id __nullable value){
    
    if (!value || [value isEqual:[NSNull null]]) {
        return nil;
    }
    
    return value;
}


void SEAnalyticsCpp::se_trackIAPWithAttributes(FSEPurchaseEventAttributes att) {
    FString logMsg = FString::Printf(
        TEXT("[trackIAPWithAttributes] called\n")
        TEXT("  productId: %s\n")
        TEXT("  productName: %s\n")
        TEXT("  productCount: %d\n")
        TEXT("  orderId: %s\n")
        TEXT("  payAmount: %f\n")
        TEXT("  currencyType: %s\n")
        TEXT("  payType: %s\n")
        TEXT("  payStatus: %d\n")
        TEXT("  failReason: %s"),
        *att.productId,
        *att.productName,
        att.productCount,
        *att.orderId,
        att.payAmount,
        *att.currencyType,
        *att.payType,
        (int)att.payStatus,
        *att.failReason
    );
    FSELog::Info(CUR_LOG_POSITION, logMsg);

    SEIAPEventAttribute *attribute = [[SEIAPEventAttribute alloc] init];
    attribute.productID    = FStringToNSString(att.productId);
    attribute.productName  = FStringToNSString(att.productName);
    attribute.productCount = att.productCount;
    attribute.orderId      = FStringToNSString(att.orderId);
    
    double payAmount = att.payAmount;
    NSDecimalNumber *d = [NSDecimalNumber decimalNumberWithString:[NSString stringWithFormat:@"%g", payAmount]];
    
    attribute.payAmount    =  [d doubleValue];;
    attribute.currencyType = FStringToNSString(att.currencyType);
    attribute.payType      = FStringToNSString(att.payType);
    attribute.payStatus    = (SolarEngineIAPStatus) att.payStatus;
    attribute.failReason   = FStringToNSString(att.failReason);

    if (att.customProperties.IsValid()) {
        attribute.customProperties = convertFJsonObjectToNSDictionary(att.customProperties);
    }

    [[SolarEngineSDK sharedInstance] trackIAPWithAttributes:attribute];
}

void SEAnalyticsCpp::se_trackAdImpressionWithAttributes(FImpressionAttributes att) {
    FString logMsg = FString::Printf(
        TEXT("[trackAdImpressionWithAttributes] called\n")
        TEXT("  adPlatform: %s\n")
        TEXT("  mediationPlatform: %s\n")
        TEXT("  adAppId: %s\n")
        TEXT("  adId: %s\n")
        TEXT("  adType: %d\n")
        TEXT("  adECPM: %f\n")
        TEXT("  currencyType: %s\n")
        TEXT("  isRendered: %d"),
        *att.adPlatform,
        *att.mediationPlatform,
        *att.adAppId,
        *att.adId,
        att.adType,
        att.adECPM,
        *att.currencyType,
        att.isRendered
    );
    FSELog::Info(CUR_LOG_POSITION, logMsg);
    SEAdImpressionEventAttribute *attribute = [[SEAdImpressionEventAttribute alloc] init];
    attribute.adNetworkPlatform   = FStringToNSString(att.adPlatform);
    attribute.mediationPlatform   = FStringToNSString(att.mediationPlatform);
    attribute.adNetworkAppID      = FStringToNSString(att.adAppId);
    attribute.adNetworkPlacementID= FStringToNSString(att.adId);
    attribute.adType              = att.adType;
    attribute.ecpm                = att.adECPM;
    attribute.currency            = FStringToNSString(att.currencyType);
    attribute.rendered            = att.isRendered;
    // customProperties 转 NSDictionary
    if (att.customProperties.IsValid()) {
        attribute.customProperties = convertFJsonObjectToNSDictionary(att.customProperties);
    }
    [[SolarEngineSDK sharedInstance] trackAdImpressionWithAttributes:attribute];
}

void SEAnalyticsCpp::se_trackAdClickWithAttributes(FAdClickAttributes att) {
    FString logMsg = FString::Printf(
        TEXT("[trackAdClickWithAttributes] called\n")
        TEXT("  adNetworkPlatform: %s\n")
        TEXT("  adType: %d\n")
        TEXT("  adNetworkPlacementId: %s\n")
        TEXT("  mediationPlatform: %s"),
        *att.adNetworkPlatform,
        (int)att.adType,
        *att.adNetworkPlacementId,
        *att.mediationPlatform
    );
    FSELog::Info(CUR_LOG_POSITION, logMsg);
    SEAdClickEventAttribute *attribute = [[SEAdClickEventAttribute alloc] init];
    attribute.adNetworkPlatform       = FStringToNSString(att.adNetworkPlatform);
    attribute.adType           = static_cast<int>(att.adType);
    attribute.adNetworkPlacementID             = FStringToNSString(att.adNetworkPlacementId);
    attribute.mediationPlatform = FStringToNSString(att.mediationPlatform);
    // customProperties 转 NSDictionary
    if (att.customProperties.IsValid()) {
        attribute.customProperties = convertFJsonObjectToNSDictionary(att.customProperties);
    }
    [[SolarEngineSDK sharedInstance] trackAdClickWithAttributes:attribute];
}

void SEAnalyticsCpp::se_trackOrderWithAttributes(FOrderEventAttributes att) {

    FString logMsg = FString::Printf(
        TEXT("[trackOrderWithAttributes] called\n")
        TEXT("  orderId: %s\n")
        TEXT("  payAmount: %f\n")
        TEXT("  currencyType: %s\n")
        TEXT("  payType: %s\n")
        TEXT("  status: %s"),
        *att.orderId,
                                     att.payAmount,
        *att.currencyType,
        *att.payType,
        *att.status
    );
    FSELog::Info(CUR_LOG_POSITION, logMsg);
    SEOrderEventAttribute *attribute = [[SEOrderEventAttribute alloc] init];
    attribute.orderID       = FStringToNSString(att.orderId);

    attribute.payAmount    = att.payAmount;
    attribute.currencyType  = FStringToNSString(att.currencyType);
    attribute.payType       = FStringToNSString(att.payType);
    attribute.status        = FStringToNSString(att.status);
    // customProperties 转 NSDictionary
    if (att.customProperties.IsValid()) {
        attribute.customProperties = convertFJsonObjectToNSDictionary(att.customProperties);
    }
    [[SolarEngineSDK sharedInstance] trackOrderWithAttributes:attribute];
}

void SEAnalyticsCpp::se_trackRegisterWithAttributes(FRegisterEventAttributes att) {
    FString logMsg = FString::Printf(
        TEXT("[trackRegisterWithAttributes] called\n")
        TEXT("  registerType: %s\n")
        TEXT("  registerStatus: %s"),
        *att.registerType,
        *att.registerStatus
    );
    FSELog::Info(CUR_LOG_POSITION, logMsg);
    SERegisterEventAttribute *attribute = [[SERegisterEventAttribute alloc] init];
    attribute.registerType   = FStringToNSString(att.registerType);
    attribute.registerStatus = FStringToNSString(att.registerStatus);
    // customProperties 转 NSDictionary
    if (att.customProperties.IsValid()) {
        attribute.customProperties = convertFJsonObjectToNSDictionary(att.customProperties);
    }
    [[SolarEngineSDK sharedInstance] trackRegisterWithAttributes:attribute];
}

void SEAnalyticsCpp::se_trackLoginWithAttributes(FLoginEventAttributes att) {
    FString logMsg = FString::Printf(
        TEXT("[trackLoginWithAttributes] called\n")
        TEXT("  loginType: %s\n")
        TEXT("  loginStatus: %s"),
        *att.loginType,
        *att.loginStatus
    );
    FSELog::Info(CUR_LOG_POSITION, logMsg);
    SELoginEventAttribute *attribute = [[SELoginEventAttribute alloc] init];
    attribute.loginType    = FStringToNSString(att.loginType);
    attribute.loginStatus  = FStringToNSString(att.loginStatus);
    // customProperties 转 NSDictionary
    if (att.customProperties.IsValid()) {
        attribute.customProperties = convertFJsonObjectToNSDictionary(att.customProperties);
    }
    [[SolarEngineSDK sharedInstance] trackLoginWithAttributes:attribute];
}
void SEAnalyticsCpp::se_trackAppAttrWithAttributes(FSEAttAttributes att) {
    FString logMsg = FString::Printf(
        TEXT("[trackAppAttrWithAttributes] called\n")
        TEXT("  adNetwork: %s\n")
        TEXT("  subChannel: %s\n")
        TEXT("  adAccountId: %s\n")
        TEXT("  adAccountName: %s\n")
        TEXT("  adCampaignId: %s\n")
        TEXT("  adCampaignName: %s\n")
        TEXT("  adOfferId: %s\n")
        TEXT("  adOfferName: %s\n")
        TEXT("  adCreativeId: %s\n")
        TEXT("  adCreativeName: %s\n")
        TEXT("  attributionPlatform: %s"),
        *att.adNetwork,
        *att.subChannel,
        *att.adAccountId,
        *att.adAccountName,
        *att.adCampaignId,
        *att.adCampaignName,
        *att.adOfferId,
        *att.adOfferName,
        *att.adCreativeId,
        *att.adCreativeName,
        *att.attributionPlatform
    );
    FSELog::Info(CUR_LOG_POSITION, logMsg);
    SEAppAttrEventAttribute *attribute = [[SEAppAttrEventAttribute alloc] init];
    attribute.adNetwork           = FStringToNSString(att.adNetwork);
    attribute.subChannel          = FStringToNSString(att.subChannel);
    attribute.adAccountID         = FStringToNSString(att.adAccountId);
    attribute.adAccountName       = FStringToNSString(att.adAccountName);
    attribute.adCampaignID        = FStringToNSString(att.adCampaignId);
    attribute.adCampaignName      = FStringToNSString(att.adCampaignName);
    attribute.adOfferID           = FStringToNSString(att.adOfferId);
    attribute.adOfferName         = FStringToNSString(att.adOfferName);
    attribute.adCreativeID        = FStringToNSString(att.adCreativeId);
    attribute.adCreativeName      = FStringToNSString(att.adCreativeName);
    attribute.attributionPlatform = FStringToNSString(att.attributionPlatform);
    // customProperties 转 NSDictionary
    if (att.customProperties.IsValid()) {
        attribute.customProperties = convertFJsonObjectToNSDictionary(att.customProperties);
    }
    [[SolarEngineSDK sharedInstance] trackAppAttrWithAttributes:attribute];
}

void SEAnalyticsCpp::se_trackAppReEngagement(FString Properties){
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[trackAppReEngagement] called, Properties: %s"), *Properties));
    NSDictionary *dict = convertToDictionary(TCHAR_TO_UTF8(*Properties));
    [[SolarEngineSDK sharedInstance] trackAppReEngagement:dict];
    
}

void SEAnalyticsCpp::se_eventStart(FString eventName) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[eventStart] called, eventName: %s"), *eventName));
    NSString *eventNameStr =(*eventName != nullptr && !eventName.IsEmpty()) ? [NSString stringWithUTF8String:TCHAR_TO_UTF8(*eventName)] : nil;
    [[SolarEngineSDK sharedInstance] eventStart:eventNameStr];
}

void SEAnalyticsCpp::se_eventFinish(FString eventName, FString propertiesJson) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[eventFinish] called, eventName: %s, propertiesJson: %s"), *eventName, *propertiesJson));
    NSString *eventNameStr =(*eventName != nullptr && !eventName.IsEmpty()) ? [NSString stringWithUTF8String:TCHAR_TO_UTF8(*eventName)] : nil;
    NSData *data = [[NSString stringWithUTF8String:TCHAR_TO_UTF8(*propertiesJson)] dataUsingEncoding:NSUTF8StringEncoding];
    NSDictionary *dict = nil;
    if (data) {
        dict = [NSJSONSerialization JSONObjectWithData:data options:0 error:nil];
        if (![dict isKindOfClass:[NSDictionary class]]) { dict = nil; }
    }
    [[SolarEngineSDK sharedInstance] eventFinish:eventNameStr properties:dict];
}

void SEAnalyticsCpp::se_reportEventImmediately() {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[reportEventImmediately] called"));
    [[SolarEngineSDK sharedInstance] reportEventImmediately];
}


TSharedPtr<FJsonObject> SEAnalyticsCpp::se_getAttributionData() {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[getAttributionData] called"));
    NSDictionary* attributionData = [[SolarEngineSDK sharedInstance] getAttributionData];
   
    return convertNSDictionaryToJsonObject(attributionData);
}

TSharedPtr<FJsonObject> SEAnalyticsCpp::se_getPresetProperties() {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[getPresetProperties] called"));
    NSDictionary *presetProperties = [[SolarEngineSDK sharedInstance] getPresetProperties];
    if (!presetProperties) return MakeShared<FJsonObject>();
    TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
    for (id Key in presetProperties) {
        id Value = presetProperties[Key];
        FString KeyStr = UTF8_TO_TCHAR([[Key description] UTF8String]);
        FString ValueStr = UTF8_TO_TCHAR([[Value description] UTF8String]);
        JsonObject->SetStringField(KeyStr, ValueStr);
    }
    return JsonObject;
}

// SEAnalytics.cpp





void SEAnalyticsCpp::se_setAttributionCallback(const FAttributionCallback&callback) {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[setAttributionCallback] called"));
    
    se_attributionCallback = callback;

    [[SolarEngineSDK sharedInstance] setAttributionCallback:^(int code, NSDictionary * _Nullable attributionData) {
        FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[setAttributionCallback] callback triggered, code: %d"), code));
        // 创建 JsonObject 包装结构
        TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
        if (attributionData) {
            for (NSString *key in attributionData.allKeys) {
                id value = attributionData[key];
                FString KeyString = UTF8_TO_TCHAR([key UTF8String]);
                FString ValueString = UTF8_TO_TCHAR([[value description] UTF8String]);
                JsonObject->SetStringField(KeyString, ValueString);
            }
        }
        // 创建包装结构
        FSEAttributionData attributionDataWrapper;
        attributionDataWrapper.code=code;
        attributionDataWrapper.data = JsonObject;
        AsyncTask(ENamedThreads::GameThread, [=]() {
                         if (se_attributionCallback.IsBound())
                         {
                             AsyncTask(ENamedThreads::GameThread, [attributionDataWrapper]() mutable {
                                  se_attributionCallback.Execute(attributionDataWrapper);
                              });                         }
                     });
                
    }];
}
void SEAnalyticsCpp::se_setInitCompletedCallback(const FInitCompletedCallback& callback)
{
    FSELog::Info(CUR_LOG_POSITION, TEXT("[setInitCompletedCallback] called"));
    se_initCompletedCallback=callback;
    [[SolarEngineSDK sharedInstance] setInitCompletedCallback:^(int code) {
        FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[setInitCompletedCallback] callback triggered, code: %d"), code));
    
        if (se_initCompletedCallback.IsBound())
        {
            AsyncTask(ENamedThreads::GameThread, [code]() mutable {
                  se_initCompletedCallback.Execute(code);
              });        }
    }];
}

void SEAnalyticsCpp::se_requestTrackingAuthorization(const FonRequestTrackingAuthorizationCallback & callback) {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[requestTrackingAuthorization] called"));
    se_onRequestTrackingAuthorizationCallback=callback;
[[SolarEngineSDK sharedInstance] requestTrackingAuthorizationWithCompletionHandler:^(NSUInteger status) {

if (se_onRequestTrackingAuthorizationCallback.IsBound()) {
    AsyncTask(ENamedThreads::GameThread, [status]() mutable {
         se_onRequestTrackingAuthorizationCallback.Execute(status);
     });}
}];
}

void SEAnalyticsCpp::se_setDeferredDeeplinkCallback( const FDeferredDeeplinkCallBack&callback) {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[setDeferredDeeplinkCallback] called"));
    se_deferredDeeplinkCallback=callback;
    [[SolarEngineSDK sharedInstance] setDeferredDeepLinkCallbackWithSuccess:^(SEDeferredDeeplinkInfo * _Nullable deeplinkInfo) {
        FDeferredDeeplinkData deeplink;
        
        deeplink.code = 0;
                   deeplink.turlId = NSStringToFString(deeplinkInfo.turlId);
                   deeplink.sedpLink = NSStringToFString(deeplinkInfo.sedpLink);
                   deeplink.sedpUrlscheme = NSStringToFString(deeplinkInfo.sedpUrlscheme);
        if (se_deferredDeeplinkCallback.IsBound()) {
            
            AsyncTask(ENamedThreads::GameThread, [deeplink]() mutable {
                se_deferredDeeplinkCallback.Execute(deeplink);
            });
        }
    }fail:^(NSError * _Nullable error) {
        
        FDeferredDeeplinkData deeplink;
        deeplink.code = (int)error.code;
        if (se_deferredDeeplinkCallback.IsBound()) {
            
            AsyncTask(ENamedThreads::GameThread, [deeplink]() mutable {
                se_deferredDeeplinkCallback.Execute(deeplink);
            });
        }
    }];
}
void SEAnalyticsCpp::se_setDeepLinkCallBack(const FDeeplinkCallBack& callback) {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[setDeepLinkCallBack] called"));
    se_deeplinkCallback = callback;
[[SolarEngineSDK sharedInstance] setDeepLinkCallback:^(int code, SEDeeplinkInfo * _Nullable deeplinkInfo) {


FDeeplinkData deeplink;
deeplink.code = code;
if (code == 0) {
deeplink.turlId = NSStringToFString(deeplinkInfo.turlId);
deeplink.sedpLink = NSStringToFString(deeplinkInfo.sedpLink);
deeplink.from = NSStringToFString(deeplinkInfo.from);
deeplink.baseUrl = NSStringToFString(deeplinkInfo.baseUrl);
deeplink.url = NSStringToFString(deeplinkInfo.url);
deeplink.customParams = convertNSDictionaryToJsonObject(deeplinkInfo.customParams);
}

if (se_deeplinkCallback.IsBound()) {

    AsyncTask(ENamedThreads::GameThread, [deeplink]() mutable {
        se_deeplinkCallback.Execute(deeplink);
    });
}

}];
}


void SEAnalyticsCpp::se_appDeeplinkOpenURL(FString url)
{
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[appDeeplinkOpenURL] called, url: %s"), *url));

    NSString* _url = FStringToNSString(url);
    NSURL* nsUrl = [NSURL URLWithString:_url];
    [[SolarEngineSDK sharedInstance] appDeeplinkOpenURL:nsUrl];
}

//在线参数

// 设置 RemoteConfig 事件属性

void SEAnalyticsCpp::se_setRemoteConfigEventProperties(FString Properties) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[setRemoteConfigEventProperties] called, Properties: %s"), *Properties));
    NSDictionary *dict = convertToDictionary(TCHAR_TO_UTF8(*Properties));
    [[SESDKRemoteConfig sharedInstance] setRemoteConfigEventProperties:dict];
}

void SEAnalyticsCpp::se_setRemoteConfigUserProperties(FString Properties) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[setRemoteConfigUserProperties] called, Properties: %s"), *Properties));
    NSDictionary *dict = convertToDictionary(TCHAR_TO_UTF8(*Properties));
    [[SESDKRemoteConfig sharedInstance] setRemoteConfigUserProperties:dict];
}

void SEAnalyticsCpp::se_setRemoteDefaultConfig(FString config)
{
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[setRemoteDefaultConfig] called, config: %s"), *config));
    NSArray *defaultConfig = nil;
    NSString *_config = FStringToNSString(config);
    if (![_config isEqualToString:@"null"]) {
        NSData *data = [_config dataUsingEncoding:NSUTF8StringEncoding];
        NSError *error = nil;
        defaultConfig = [NSJSONSerialization JSONObjectWithData:data options:0 error:&error];
        if (error) {
            NSString *msg = [NSString stringWithFormat:@"%@ is not an invalid JSON data",_config];
            NSLog(@"__iOSSESDKSetRemoteConfigUserProperties, error :%@",msg);
            return;
        }
    }
    [[SESDKRemoteConfig sharedInstance] setDefaultConfig:defaultConfig];
}
FString SEAnalyticsCpp::se_fastFetchRemoteConfig(const FString& key) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[fastFetchRemoteConfig] called, key: %s"), *key));
    NSString *keyStr = FStringToNSString(key);
    id result = [[SESDKRemoteConfig sharedInstance] fastFetchRemoteConfig:keyStr];
    if (!result) return FString("");
    FString ValueStr = ConvertNSObjectToFString(result);
    return ValueStr;
}
TSharedPtr<FJsonObject> SEAnalyticsCpp::se_fastAllFetchRemoteConfig() {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[fastAllFetchRemoteConfig] called"));
    NSDictionary *config = [[SESDKRemoteConfig sharedInstance] fastFetchRemoteConfig];
    TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
    if (config) {
        for (id key in config) {
            id value = config[key];  // ✅ 先取出 value
            FString KeyStr = UTF8_TO_TCHAR([[key description] UTF8String]);
            FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[fastAllFetchRemoteConfig] key=%s"), *KeyStr));
            JsonObject->SetField(KeyStr, ConvertNSObjectToFJsonValue(value));
        }
    }
    return JsonObject;
}

void SEAnalyticsCpp::se_asyncFetchRemoteConfig(const FString& key, FonFetchRemoteConfigCallback callback) {
    FSELog::Info(CUR_LOG_POSITION, FString::Printf(TEXT("[asyncFetchRemoteConfig] called, key: %s"), *key));
    NSString *keyStr = FStringToNSString(key);
    
    se_onFetchRemoteConfigCallback=callback;
    [[SESDKRemoteConfig sharedInstance] asyncFetchRemoteConfigWithCompletionHandler:^(NSDictionary * _Nullable config) {
        id result = config[keyStr];
        FString ValueStr = ConvertNSObjectToFString(result);
        FString::Printf(TEXT("value=%s"), SafeStr(ValueStr));
          if (se_onFetchRemoteConfigCallback.IsBound()) {
            AsyncTask(ENamedThreads::GameThread, [callback, ValueStr]() mutable {
                se_onFetchRemoteConfigCallback.Execute(ValueStr);
            });
        }
    }];
}
void SEAnalyticsCpp::se_asyncAllFetchRemoteConfig(FonFetchAllRemoteConfigCallback callback) {
    FSELog::Info(CUR_LOG_POSITION, TEXT("[asyncAllFetchRemoteConfig] called"));
    se_onFetchAllRemoteConfigCallback=callback;
    [[SESDKRemoteConfig sharedInstance] asyncFetchRemoteConfigWithCompletionHandler:^(NSDictionary * _Nullable config) {
        FSELog::Info(CUR_LOG_POSITION, TEXT("[asyncAllFetchRemoteConfig] callback triggered"));
        TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
        if (config) {
            for (id key in config) {
                id value = config[key];  // ✅ 先取出 value
                FString KeyStr = UTF8_TO_TCHAR([[key description] UTF8String]);

                FSELog::Info(
                    CUR_LOG_POSITION,
                    FString::Printf(
                        TEXT("[asyncAllFetchRemoteConfig] key=%s"),
                        *KeyStr
                    )
                );

                JsonObject->SetField(KeyStr, ConvertNSObjectToFJsonValue(value));
            }
        }
        FSERemoteConfigData dataWrapper;
        dataWrapper.data = JsonObject;
        if (se_onFetchAllRemoteConfigCallback.IsBound()) {
            AsyncTask(ENamedThreads::GameThread, [callback, dataWrapper]() mutable {
                se_onFetchAllRemoteConfigCallback.Execute(dataWrapper);
            });
        }
    }];
}
FString SEAnalyticsCpp::ConvertNSObjectToFString(id result)
{
    if (!result || result == [NSNull null]) {
        return FString("");
    }
    if ([result isKindOfClass:[NSString class]]) {
        return NSStringToFString((NSString *)result);
    }
    else if ([result isKindOfClass:[NSNumber class]]) {
        NSNumber *num = (NSNumber *)result;
        if (strcmp([num objCType], @encode(BOOL)) == 0) {
            return FString(num.boolValue ? TEXT("true") : TEXT("false"));
        }
        return NSStringToFString([num stringValue]);
    }
    else if ([result isKindOfClass:[NSDictionary class]] || [result isKindOfClass:[NSArray class]]) {
        NSError *error = nil;
        NSData *jsonData = [NSJSONSerialization dataWithJSONObject:result options:0 error:&error];
        if (jsonData && !error) {
            NSString *jsonString = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
            if (jsonString) {
                return NSStringToFString(jsonString);
            }
        }
        NSLog(@"[SEAnalyticsCpp] JSON serialization failed: %@", error);
        return FString("");
    }
    return FString("");
}
TSharedPtr<FJsonValue> SEAnalyticsCpp::ConvertNSObjectToFJsonValue(id value)
{
    if ([value isKindOfClass:[NSString class]]) {
        return MakeShared<FJsonValueString>(UTF8_TO_TCHAR([(NSString *)value UTF8String]));
    }
    else if ([value isKindOfClass:[NSNumber class]]) {
        NSNumber *num = (NSNumber *)value;
        const char *type = [num objCType];

        if (strcmp(type, @encode(BOOL)) == 0 || strcmp(type, "c") == 0) {
            return MakeShared<FJsonValueBoolean>(num.boolValue);
        }

        return MakeShared<FJsonValueNumber>([num doubleValue]);
    }
    else if ([value isKindOfClass:[NSDictionary class]]) {
        NSError *error = nil;
        NSData *jsonData = [NSJSONSerialization dataWithJSONObject:value options:0 error:&error];
        if (jsonData && !error) {
            NSString *jsonStr = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
            FString JsonFStr = UTF8_TO_TCHAR([jsonStr UTF8String]);
            TSharedPtr<FJsonObject> Obj;
            TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(JsonFStr);
            if (FJsonSerializer::Deserialize(Reader, Obj) && Obj.IsValid()) {
                return MakeShared<FJsonValueObject>(Obj);
            }
        }
    }
    else if ([value isKindOfClass:[NSArray class]]) {
        NSError *error = nil;
        NSData *jsonData = [NSJSONSerialization dataWithJSONObject:value options:0 error:&error];
        if (jsonData && !error) {
            NSString *jsonStr = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
            FString JsonFStr = UTF8_TO_TCHAR([jsonStr UTF8String]);
            TArray<TSharedPtr<FJsonValue>> JsonArray;
            TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(JsonFStr);
            if (FJsonSerializer::Deserialize(Reader, JsonArray)) {
                return MakeShared<FJsonValueArray>(JsonArray);
            }
        }
    }
    else if ([value isKindOfClass:[NSNull class]]) {
        return MakeShared<FJsonValueNull>();
    }
    return MakeShared<FJsonValueNull>();
}





#endif

 
