#pragma once
#include "CoreMinimal.h"
#include "UObject/Object.h"
#include "Templates/SharedPointer.h"  //
#include <map>
#include <string>
#include <vector>
#include "SEAnalytics.generated.h"


const FString SE_UE_LIB_VERSION = "1.0.0.0";

USTRUCT(BlueprintType)
struct FSEAttributionData
{
	GENERATED_BODY()
    int32 code;
	TSharedPtr<FJsonObject> data;
};
using SEObject = TVariant<
	FString,
	int,
	bool,
	TSharedPtr<FJsonObject>// 支持嵌套字典
>;
USTRUCT(BlueprintType)
struct FSERemoteConfigData
{
    GENERATED_BODY()
    
    TSharedPtr<FJsonObject> data;
};
USTRUCT(BlueprintType)

struct FDeferredDeeplinkData
{
    GENERATED_BODY()
    int code = -1;

    FString sedpLink;
    FString turlId;
    FString sedpUrlscheme;
};
USTRUCT(BlueprintType)
struct FDeeplinkData
{
    GENERATED_BODY()
	int code = -1;

	FString sedpLink;
	FString turlId;
	FString from;
	FString baseUrl;
	FString url;
	TSharedPtr<FJsonObject>  customParams;
};

// 归因回调委托（使用包装结构，避免 UHT 宏参数中模板逗号问题）
DECLARE_DELEGATE_OneParam(FAttributionCallback, FSEAttributionData);
DECLARE_DELEGATE_OneParam(FInitCompletedCallback, int32);

DECLARE_DELEGATE_OneParam(FonRequestTrackingAuthorizationCallback, int32);
DECLARE_DELEGATE_OneParam(FDeferredDeeplinkCallBack, FDeferredDeeplinkData);
DECLARE_DELEGATE_OneParam(FDeeplinkCallBack, FDeeplinkData);

DECLARE_DELEGATE_OneParam(FonFetchRemoteConfigCallback, FString);
DECLARE_DELEGATE_OneParam(FonFetchAllRemoteConfigCallback, FSERemoteConfigData);



UENUM(BlueprintType)
enum class ESEPresetEventType : uint8
{
	AppInstall,
	AppStart,
	AppEnd,
	All
};

UENUM(BlueprintType)
enum class ESEFirstEventType : uint8
{
	Unknown,
	Register,
	Custom
};
UENUM(BlueprintType)
enum class ESEPayStatus : uint8
{
	None = 0,
	Success = 1,
	Failed = 2
};
UENUM(BlueprintType)
enum class ESEMergeType : uint8
{
	ByUser = 0,
    ByDefault = 1,
	
};


// Step 1: 定义通用类型 Object（模拟 C# 的 object）


// 定义 Item 结构
struct Item {
	FString name;
	SEObject value;  // 用 TVariant 存不同类型
	int type;        // 自己定义的类型码
};

namespace SEAnalytics
{
	inline Item stringItem(const FString& name, const FString& value) {
		// std::string -> FString
		return {name, SEObject(TInPlaceType<FString>(), value), 1};
	}	

	inline Item intItem(const FString& name, int value) {
		return {name, SEObject(TInPlaceType<int>(), value), 2};
	}

	inline Item boolItem(const FString& name, bool value) {
		return {name, SEObject(TInPlaceType<bool>(), value), 3};
	}

	inline Item jsonItem(const FString& name, const TSharedPtr<FJsonObject>& value) {
		return {name, SEObject(TInPlaceType<TSharedPtr<FJsonObject>>(), value), 4};
	}
}


USTRUCT(BlueprintType)
struct FSECCustomDomain {
	GENERATED_BODY()
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
	bool enable = false;
	
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
	FString receiverDomain;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
	FString ruleDomain;
	
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
	FString receiverTcpHost;
	
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
	FString ruleTcpHost;
	
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
	FString gatewayTcpHost;
};
USTRUCT(BlueprintType)

struct FSECRemoteConfig {
	GENERATED_BODY()
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
	bool enable = false;

    TSharedPtr<FJsonObject> customIDProperties;
    TSharedPtr<FJsonObject> customIDEventProperties;
    TSharedPtr<FJsonObject> customIDUserProperties;
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
	ESEMergeType mergeType=ESEMergeType::ByDefault;
	FString ToJsonString() const;

	
};




USTRUCT(BlueprintType)
struct FSEConfig
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Log Enabled"))
	bool logEnabled = false;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Enable 2G Reporting"))
	bool enable2GReporting = false;


	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Is Debug Model"))
	bool isDebugModel = false;


	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Is GDPR Area"))
	bool isGDPRArea = false;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Set COPPA Enabled"))
	bool isCoppaEnabled = false;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Set Kids App Enabled"))
	bool isKidsAppEnabled = false;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Set Ad Personalization Enabled"))
	bool adPersonalizationEnabled = false;
	
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Set Ad User Data Enabled"))
	bool adUserDataEnabled = false;
    
    UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
    bool enableODMInfo = false;
	

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "ATT Authorization Waiting Interval"))
	int32 attAuthorizationWaitingInterval = 0;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Caid"))
	FString caid;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "FB App ID"))
	FString fbAppId;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Enable Delay Deeplink"))
	bool deferredDeeplinkenable = false;



	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Enable Multi Process"))
	bool supportMultiProcess = false;
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Is OAID Enabled"))
	bool isOAIdEnabled = false;
	
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Is Imei Enabled"))
	bool isImeiEnabled = false;
	
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Is Android ID Enabled"))
	bool isAndroidIdEnabled = false;
	
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Enable IPV6 Address"))
	bool enableIPV6Address = false;

	// UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "With Disable OAID Retry"))
	// bool withDisableOAIDRetry = false;
	//
	// UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "With Disable GAID Retry"))
	// bool withDisableGAIDRetry = false;

	FAttributionCallback attributionCallback;

	FInitCompletedCallback initCompletedCallback;
	
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics")
	FSECCustomDomain customDomain;

// 	
// private:
	private:
	UPROPERTY(SaveGame, meta = (AllowPrivateAccess = "true"))
	FString subLibVersion = SE_UE_LIB_VERSION;

};


USTRUCT(BlueprintType)
struct FSEBaseAttributes
{
	GENERATED_BODY()
	FSEBaseAttributes() : Type(ESEFirstEventType::Unknown) {}

	// 首次事件校验id
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Check ID"))
	FString checkId;

	// 提供外部只读访问
	FORCEINLINE ESEFirstEventType GetType() const { return Type; }
protected:
	ESEFirstEventType Type;
	
};

USTRUCT(BlueprintType)
struct FSEPurchaseEventAttributes
{
	GENERATED_BODY()

	// 商品 ID
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Product ID"))
	FString productId;

	// 商品名称
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Product Name"))
	FString productName;

	// 商品数量
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Product Count"))
	int32 productCount = 1;

	// 订单 ID
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Order ID"))
	FString orderId;

	// 支付金额（单位：元）
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Pay Amount"))
	double payAmount = 0.0;

	// 货币类型（如 CNY, USD）
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Currency Type"))
	FString currencyType;

	// 支付方式（alipay, weixin, applepay, paypal）
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Pay Type"))
	FString payType;

	// 支付状态
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Pay Status"))
	int payStatus;

	// 支付失败原因
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Fail Reason"))
	FString failReason;

	// 自定义属性
	//UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Custom Properties"))
	TSharedPtr<FJsonObject> customProperties;
};


USTRUCT(BlueprintType)
struct FImpressionAttributes
{
	GENERATED_BODY()

	// 变现平台名称，不可为空
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Platform"))
	FString adPlatform;

	// 聚合平台名称，不可为空，否则填 "custom"
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Mediation Platform"))
	FString mediationPlatform;

	// 变现平台的应用 ID，不可为空
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad App ID"))
	FString adAppId;

	// 变现平台的广告位 ID，可为空
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad ID"))
	FString adId;

	// 展示广告的类型，不可为空（使用枚举）
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Type"))
	int adType;

	// 广告 ECPM（千次展示收入），0 或负值表示未传
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad ECPM"))
	double adECPM = 0.0;

	// 收益货币类型，遵循 ISO 4217，如 "CNY", "USD"
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Currency Type"))
	FString currencyType;

	// 广告是否渲染成功
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Is Rendered"))
	bool isRendered = true;

	// 自定义属性
	//UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Custom Properties"))
	TSharedPtr<FJsonObject> customProperties;
};

USTRUCT(BlueprintType)
struct FRegisterEventAttributes: public FSEBaseAttributes
{
	GENERATED_BODY()
	FRegisterEventAttributes()
	{
		Type = ESEFirstEventType::Register;
	}
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Register Type"))
	FString registerType;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Register Status"))
	FString registerStatus;

	//UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Custom Properties"))
	TSharedPtr<FJsonObject> customProperties;
};

USTRUCT(BlueprintType)
struct FCustomAttributes: public FSEBaseAttributes
{
	GENERATED_BODY()
	FCustomAttributes()
	{
		Type = ESEFirstEventType::Custom;
	}
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Custom Event Name"))
	FString customEventName;
	//UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Custom Properties"))
	TSharedPtr<FJsonObject> customProperties;
	//preProperties
	// UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Pre Properties"))
	//TSharedPtr<FJsonObject> preProperties;
};

USTRUCT(BlueprintType)
struct FLoginEventAttributes
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Login Type"))
	FString loginType;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Login Status"))
	FString loginStatus;

	//UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Custom Properties"))
	TSharedPtr<FJsonObject> customProperties;
};


USTRUCT(BlueprintType)
struct FAdClickAttributes
{
	GENERATED_BODY()

	// 变现平台名称，不可为空
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Platform"))
	FString adNetworkPlatform;

	// 展示广告的类型，不可为空
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Type"))
	int adType ;

	// 变现平台的广告位 ID，可为空
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad ID"))
	FString adNetworkPlacementId;

	// 聚合平台名称，不可为空，否则填 "custom"
	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Mediation Platform"))
	FString mediationPlatform;

	// 自定义属性
	//UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Custom Properties"))
	TSharedPtr<FJsonObject> customProperties;
};


USTRUCT(BlueprintType)
struct FSEAttAttributes
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Network"))
	FString adNetwork;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Sub Channel"))
	FString subChannel;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Account ID"))
	FString adAccountId;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Account Name"))
	FString adAccountName;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Campaign ID"))
	FString adCampaignId;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Campaign Name"))
	FString adCampaignName;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Offer ID"))
	FString adOfferId;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Offer Name"))
	FString adOfferName;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Creative ID"))
	FString adCreativeId;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Ad Creative Name"))
	FString adCreativeName;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Attribution Platform"))
	FString attributionPlatform;

	//UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Custom Properties"))
	TSharedPtr<FJsonObject> customProperties;
};



USTRUCT(BlueprintType)
struct FOrderEventAttributes
{
	GENERATED_BODY()

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Order ID"))
	FString orderId;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Pay Amount"))
    double payAmount;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Currency Type"))
	FString currencyType;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Pay Type"))
	FString payType;

	UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Status"))
	FString status;

	//UPROPERTY(BlueprintReadWrite, Category = "SEAnalytics", meta = (DisplayName = "Custom Properties"))
	TSharedPtr<FJsonObject> customProperties;
};


UCLASS()
class SEANALYTICS_API USEAnalytics : public UObject
{
	GENERATED_UCLASS_BODY()
private:
	void init();
	FString toJsonString(TSharedPtr<FJsonObject> jsonObject);

public:
	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void preInit(const FString& appkey);

	static void init(
		const FString& appkey,
		const FSEConfig& config = FSEConfig(),
		const FSECRemoteConfig& remoteConfig = FSECRemoteConfig());
	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static FString getDistinctId();
\
	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void login(const FString& accountID);

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static FString getAccountId();
	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void logout();

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void setVisitorID(const FString& VisitorID);

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static FString getVisitorID();

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void setOaid(const FString& oaid);
	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void setGaid(const FString& gaid);
	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void setGDPRArea(const bool& isGDPRArea);
	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void setChannel(const FString& channel);

	static void setPresetEvent(ESEPresetEventType EventType, TSharedPtr<FJsonObject> Properties);


	static void setSuperProperties(TSharedPtr<FJsonObject> Properties);

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void unsetSuperProperty(const FString& Property);

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void clearSuperProperties();

	static void track(const FString& eventName, const  TSharedPtr<FJsonObject>& customAttributes,const  TSharedPtr<FJsonObject>& preAttributes= nullptr );

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void trackPurchase(const FSEPurchaseEventAttributes& att);

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void trackAdImpression(const FImpressionAttributes& att);

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void trackAdClick(const FAdClickAttributes& att);

	
	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void trackAttribution(const FSEAttAttributes& att);

	
	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void trackRegister(const FRegisterEventAttributes& att);

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void trackLogin(const FLoginEventAttributes& att);


	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void trackOrder(const FOrderEventAttributes& att);
	

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void trackFirstEvent(const FSEBaseAttributes& att);
    
    static void trackAppReEngagement(TSharedPtr<FJsonObject> Properties);

    


	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void eventStart(const FString& EventName);

	static void eventFinish(const FString& EventName, const TSharedPtr<FJsonObject>& Properties=nullptr);

	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void reportEventImmediately();

	static void userInit(TSharedPtr<FJsonObject> Properties);

	
	static void userUpdate(TSharedPtr<FJsonObject> Properties);


	static void userAdd(TSharedPtr<FJsonObject> Properties);

	static void userUnset(const TArray<FString>& keys);


	static void userAppend(TSharedPtr<FJsonObject> Properties);


	UFUNCTION(BlueprintCallable, Category = "SEAnalytics")
	static void userDelete(int32 DeleteType);
	
	static TSharedPtr<FJsonObject> getPresetProperties();


	static TSharedPtr<FJsonObject> getAttributionData();




	//在线参数api
	
	static void setRemoteDefaultConfig(const std::vector<Item>& defaultConfig);
	static void setRemoteConfigEventProperties(TSharedPtr<FJsonObject> Properties);
	static void setRemoteConfigUserProperties(TSharedPtr<FJsonObject> Properties);
	
	static FString fastFetchRemoteConfig(const FString&key);
	static TSharedPtr<FJsonObject> fastAllFetchRemoteConfig();
	static void asyncFetchRemoteConfig( const FString&key, const FonFetchRemoteConfigCallback&  callback );
	static void asyncAllFetchRemoteConfig(const FonFetchAllRemoteConfigCallback& callback );
	
	static void setDeferredDeeplinkCallback(const FDeferredDeeplinkCallBack & callback );
	static void setDeepLinkCallback(const FDeeplinkCallBack& callback);
    static void appDeeplinkOpenURL(const FString& url);
	static void requestTrackingAuthorization(const FonRequestTrackingAuthorizationCallback& callback );

	
};

